<?php
namespace WBLS_WhistleBlower\Free;
use WP_Query;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly
class WBLS_Controller {
    public function __construct( $task ) {
        if ( strpos( $task, 'submissions_' ) === 0 ) {
            require_once WBLS_DIR . '/admin/controllers/ControllerSubmissions.php';
            new WBLS_ControllerSubmissions( $task );
        } elseif ( method_exists($this, $task) && $task !== '__construct' ) {
            $this->$task();
        }
    }

    /* New insert form functionality */
    public function wbls_add_form() {
        $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
        if ( ! wp_verify_nonce( $nonce, 'wbls_ajax_nonce' ) || ! current_user_can('manage_options') ) {
            wp_send_json_error(array(
                'message' => esc_html__( 'You are not allowed to add form', 'whistleblowing-system' ),
            ));
        }


        $form_id = isset($_POST['form_id']) ? intval($_POST['form_id']) : 0;
        if ( $form_id && get_post_type( $form_id ) !== 'wbls_form' ) {
            wp_send_json_error(array(
                'message' => esc_html__( 'Wrong Form ID.', 'whistleblowing-system' ),
            ));
        }

        $form_content = '';
        if ( isset( $_POST['form'] ) ) {
            // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitized by wp_kses below
            $form_content = wp_kses( trim( wp_unslash( $_POST['form'] ) ), WBLSLibrary::$wp_kses_form );
        }

        // Remove newlines + unwanted class
        $form_content = str_replace(
            [ "\r", "\n", "wblsform-row-edit-active" ],
            '',
            $form_content
        );

        // Remove XML garbage only if present
        if ( strpos( $form_content, '?xml' ) !== false ) {
            $form_content = preg_replace('/<!--\?xml[^>]*\?-->/', '', $form_content);
        }

        $form_title = isset($_POST['form_title']) ? sanitize_text_field(wp_unslash($_POST['form_title'])) : '';
        $my_post = array(
            'post_title'    => $form_title,
            'post_status'   => 'publish',
            'post_author'   => 1,
            'post_category' => array( 8,39 ),
            'post_type' => 'wbls_form'
        );

        // Sanitize field_options
        $field_options = array();
        if ( isset($_POST['field_options']) && is_array($_POST['field_options']) ) {
            $field_options = map_deep(wp_unslash($_POST['field_options']), 'sanitize_text_field');
            foreach ( $field_options as $key => $field_option ) {
                if ( empty($field_option) ) {
                    unset($field_options[$key]);
                }
            }
        }

        // Sanitize email_options
        $email_options = array();
        if ( isset($_POST['email_options']) && is_array($_POST['email_options']) ) {
            $email_options = map_deep(wp_unslash($_POST['email_options']), function($value) {
                return wp_kses($value, WBLSLibrary::$wp_kses_default);
            });
        }

        // Sanitize form_settings with specific field handling
        $form_settings = array();
        if ( isset($_POST['form_settings']) && is_array($_POST['form_settings']) ) {
            $form_settings = map_deep(wp_unslash($_POST['form_settings']), function($value) {
                return wp_kses($value, WBLSLibrary::$wp_kses_default);
            });

            // Specific sanitization for HTML content fields
            if( !empty($form_settings['form_header']) ) {
                $form_settings['form_header'] = wp_kses($form_settings['form_header'], WBLSLibrary::$wp_kses_default);
            }
            if( !empty($form_settings['token_header']) ) {
                $form_settings['token_header'] = wp_kses($form_settings['token_header'], WBLSLibrary::$wp_kses_default);
            }
            if( !empty($form_settings['login_header']) ) {
                $form_settings['login_header'] = wp_kses($form_settings['login_header'], WBLSLibrary::$wp_kses_default);
            }
        }

        $fieldNameLastId = isset($_POST['fieldNameLastId']) ? intval($_POST['fieldNameLastId']) : 0;

        // Read and sanitize POST data.
        $form_conditions = isset( $_POST['form_conditions'] )
            ? $this->wbls_sanitize_conditions_array( wp_unslash( $_POST['form_conditions'] ) ) // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
            : [];
        if ( ! empty( $form_conditions ) ) {

            $form_conditions = $this->wbls_clear_conditions_array( $form_conditions, $field_options );
        }

        if ( $form_id ) {
            $my_post['ID'] = $form_id;
            $insert = wp_update_post( $my_post );

            if ( $insert ) {

                $post_meta = update_post_meta( $form_id, 'wbls_field_options', $field_options, false);
                update_post_meta( $form_id, 'wbls_email_options', $email_options, false);
                update_post_meta( $form_id, 'wbls_form_settings', $form_settings, false);
                update_post_meta( $form_id, 'wbls_fieldNameLastId', $fieldNameLastId, false);
                update_post_meta( $form_id, 'wbls_form_content', $form_content, false);
                wp_update_post( array('ID' => $form_id, 'post_content' => '[wblsform id="' . $form_id . '"]') );
                $db_last_conditions = get_post_meta($form_id, 'wbls_form_conditions');
                update_post_meta($form_id, 'wbls_form_conditions', $form_conditions, false);
                $this->save_income_webhook( $form_id );
                $this->save_outgoing_webhook( $form_id );

                $reload_url = '';
                if( $post_meta ) {
                    $reload_url = add_query_arg(array(
                        'page' => 'whistleblower_form_edit',
                        'id' => $form_id,
                    ), admin_url('admin.php'));
                }


                wp_send_json_success(array(
                    'form_id' => intval($insert),
                    'message' => esc_html__('Form successfully updated.', 'whistleblowing-system'),
                    'reload_url' => $reload_url,
                ), 200);
            }
        } else {
            $insert = wp_insert_post( $my_post );
            if( $insert ) {
                $post_meta = add_post_meta( $insert, 'wbls_field_options', $field_options, true );
                add_post_meta( $insert, 'wbls_email_options', $email_options, true );
                add_post_meta( $insert, 'wbls_form_settings', $form_settings, true );
                add_post_meta( $insert, 'wbls_fieldNameLastId', $fieldNameLastId, true );
                add_post_meta( $insert, 'wbls_form_content', $form_content, true);
                $this->save_income_webhook( $insert );
                $this->save_outgoing_webhook( $insert );
                if( ! empty( $form_conditions ) ) {
                    add_post_meta($form_id, 'wbls_form_conditions', $form_conditions, true);
                    $args = [
                        'form_id' => $form_id,
                        'field_options' => $field_options,
                        'form_conditions' => $form_conditions
                    ];
                    new \WBLS_WhistleBlower\Free\WBLS_Conditions($args);
                }
                wp_update_post( array('ID' => $insert, 'post_content' => '[wblsform id="' . intval($insert) . '"]') );

                $reload_url = add_query_arg(array(
                                                'page' => 'whistleblower_form_edit',
                                                'id' => intval($insert),
                                            ), admin_url('admin.php'));

                if( $post_meta ) {
                    wp_send_json_success(array(
                        'form_id' => intval($insert),
                        'message' => esc_html__('Form successfully updated.', 'whistleblowing-system'),
                        'reload_url' => $reload_url,
                    ), 200);
                } else {
                    wp_delete_post( $insert, true);
                    wp_send_json_error(array(
                        'message' => esc_html__( 'Error during the field or email options save, please try again.', 'whistleblowing-system' ),
                    ));
                }
            }
        }
        wp_send_json_error(array(
            'message' => esc_html__( 'Something went wrong, please try again.', 'whistleblowing-system' ),
        ));
    }

    /**
     * Recursively sanitize form conditions array.
     */
    private function wbls_sanitize_conditions_array( $array ) {

        if ( ! is_array( $array ) ) {
            return [];
        }

        foreach ( $array as $key => $value ) {

            // Always sanitize keys
            $clean_key = sanitize_key( (string) $key );

            if ( is_array( $value ) ) {
                $array[ $clean_key ] = $this->wbls_sanitize_conditions_array( $value );
                continue;
            }

            // Sanitize scalar values based on purpose
            switch ( $clean_key ) {
                case 'field_id':
                case 'group_id':
                case 'condition_item_id':
                    $array[ $clean_key ] = sanitize_text_field( $value );
                    break;

                case 'value':
                case 'operator':
                    $array[ $clean_key ] = sanitize_text_field( $value );
                    break;

                default:
                    $array[ $clean_key ] = sanitize_text_field( $value );
                    break;
            }
        }

        return $array;
    }

    public function save_income_webhook( $form_id ){}

    public function save_outgoing_webhook( $form_id )  {}

    /**
     *  Clearing Conditions array empty keys and removed fields
     *
     * @params $form_conditions array which has structure condition[field_id][conditions][group_id][condition_item_id]
     *
     * @return array $form_conditions
    */
    private function wbls_clear_conditions_array( $form_conditions, $field_options ) {
        /* This is condition[field_id] foreach */
        foreach ($form_conditions as $key => $val ) {
            if( empty($val) || empty($val['conditions']) || !isset($field_options[$key]) ) {
                unset($form_conditions[$key]);
            } else {
                /* This is condition[field_id][conditions] foreach */
                foreach($val['conditions'] as $key1 => $val1 ) {
                    if( empty($val1) ) {
                        unset($form_conditions[$key]['conditions'][$key1]);
                    } else {
                        /* This is condition[field_id][group_id] foreach */
                        foreach($val1 as $key2 => $val2 ) {
                            if( empty($val2) ||
                                ( is_numeric( $val2['field_id'] ) && ! isset( $field_options[$val2['field_id']] ) )
                            ) {
                                unset($form_conditions[$key]['conditions'][$key1][$key2]);
                            }
                            elseif( ! is_numeric( $val2['field_id'] ) ) {
                                $field_name = 'wbls_field_' . $val2['field_id'];
                                $found = false;
                                array_walk_recursive($field_options, function($value) use ($field_name, &$found) {
                                    if ($value === $field_name) {
                                        $found = true;
                                    }
                                });

                                if ( ! $found ) {
                                    unset($form_conditions[$key]['conditions'][$key1][$key2]);
                                }
                            }
                        }
                    }
                }
            }
        }

        return $form_conditions;
    }

    public function wbls_reply() {}

    public function wbls_set_default_theme() {}

    public function remove_form() {
        $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
        if ( ! wp_verify_nonce( $nonce, 'wbls_ajax_nonce' ) || ! current_user_can('manage_options') ) {
            wp_send_json_error(["nonce" => 'false']);
        }

        $id = isset($_POST['id']) ? intval($_POST['id']) : 0;
        wp_delete_post($id);
        delete_post_meta($id, 'wbls_field_options');
        delete_post_meta($id, 'wbls_email_options');
        delete_post_meta($id, 'wbls_form_settings');
        delete_post_meta($id, 'wbls_fieldNameLastId');
        delete_post_meta($id, 'wbls_form_content');
        delete_post_meta($id, 'wbls_form_conditions');

        $args = array(
            'post_type'		=>	'wbls_form_subm',
            'post_status' => 'closed',
            'meta_query'	=>	array(
                array(
                    'key'	=>	'wbls_form_id',
                    'value'	=>	$id,
                )
            ),
            "numberposts" => 1000,
            "posts_per_page" => 1000,
        );
        $query = new WP_Query( $args );
        if ( $query->posts ) {
            global $wpdb;
            foreach ( $query->posts as $post ) {
                $post_id = isset( $post->ID ) ? absint( $post->ID ) : 0;

                if ( $post_id ) {
                    $wpdb->query(
                        $wpdb->prepare(
                            "DELETE p, pm
                                    FROM {$wpdb->posts} p
                                    INNER JOIN {$wpdb->postmeta} pm ON pm.post_id = p.ID
                                    WHERE p.ID = %d",
                            $post_id
                        )
                    );
                }
            }
        }
        $current_user = wp_get_current_user();
        $created_by = 'admin_na';
        // The standard, most reliable check
        if ( is_user_logged_in() ) {
            $username = $current_user->user_login;
            $created_by = $username;
        }

        \WBLS_WhistleBlower\Free\WBLS_Logger::log( 'form_deleted',
            'success',
            'The form with ID ' . $id . ' and all its data, including submissions, have been deleted.',
            [
                'form_id' => $id,
                'submission_id' => 0,
                'created_by' => $created_by,
            ]);

        wp_send_json_success();
    }

    public function remove_theme() {
        $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
        if ( ! wp_verify_nonce( $nonce, 'wbls_ajax_nonce' ) || ! current_user_can('manage_options') ) {
            wp_send_json_error(["nonce" => 'false']);
        }

        $id = isset($_POST['id']) ? intval($_POST['id']) : 0;

        wp_delete_post($id);
        delete_post_meta($id, 'wbls_theme');
        $wp_upload_dir = wp_upload_dir();
        $wbls_style_file = $wp_upload_dir[ 'basedir' ] . '/wbls-system/wbls-theme-style_'.$id.'.css';
        unlink($wbls_style_file);
        wp_send_json_success();
    }

    public function wbls_save_settings() {
        $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
        if ( ! wp_verify_nonce( $nonce, 'wbls_ajax_nonce' ) || ! current_user_can('manage_options') ) {
            wp_send_json_error(["nonce" => 'false']);
        }
        $teeny_active = isset($_POST['teeny_active']) ? intval($_POST['teeny_active']) : 1;
        update_option( 'teeny_active', $teeny_active );
        $data = [];
        $data['user_token_visibility_active'] = isset($_POST['user_token_visibility_active']) ? intval($_POST['user_token_visibility_active']) : 1;
        $data['logs_active'] = isset($_POST['logs_active']) ? intval($_POST['logs_active']) : 0;
        $data['logs_lifetime'] = isset($_POST['logs_lifetime']) ? intval($_POST['logs_lifetime']) : 30;

        update_option( 'wbls_global_settings', wp_json_encode($data) );
        wp_send_json_success();

    }
}
