<?php
namespace WBLS_WhistleBlower\Free;

/**
 * Rate Notice for Whistleblowing System
*/
if ( ! defined('ABSPATH') ) exit;

class WBLS_Rate_Notice {

    const INSTALL_DATE_KEY  = 'wbls_install_date';
    const DISMISSED_KEY     = 'wbls_rate_notice_dismissed';
    const REMIND_KEY        = 'wbls_rate_notice_remind';
    const SHOW_AFTER_DAYS   = 10; // first time after 10 days
    const REMIND_AFTER_DAYS = 7;  // show again after 7 days

    public function __construct() {
        add_action('admin_notices', [$this, 'maybe_show_notice']);
        add_action('admin_init', [$this, 'handle_actions']);
    }

    public static function set_install_date() {
        if ( ! get_option(self::INSTALL_DATE_KEY) ) {
            update_option(self::INSTALL_DATE_KEY, time());
        }
    }

    public static function maybe_show_notice() {
        if ( ! current_user_can('manage_options') ) {
            return;
        }

        $install_date = get_option(self::INSTALL_DATE_KEY);
        $dismissed    = get_option(self::DISMISSED_KEY);
        $remind_until = get_option(self::REMIND_KEY);

        // If never set (old users) → start clock now
        if ( ! $install_date ) {
            update_option(self::INSTALL_DATE_KEY, time());
            return;
        }

        // Permanently dismissed
        if ( $dismissed ) return;

        // Waiting for remind period
        if ( $remind_until && time() < $remind_until ) return;

        // Not enough days yet
        if ( time() - $install_date < DAY_IN_SECONDS * self::SHOW_AFTER_DAYS ) return;

        $rate_url    = 'https://wordpress.org/support/plugin/whistleblowing-system/reviews/#new-post';
        $remind_url  = wp_nonce_url(add_query_arg('wbls_rate_action', 'remind'), 'wbls_rate_action');
        $dismiss_url = wp_nonce_url(add_query_arg('wbls_rate_action', 'dismiss'), 'wbls_rate_action');

        $page = isset($_GET['page']) ? sanitize_key($_GET['page']) : '';

        $class = '';
        if( strpos($page, 'whistleblower_') === 0 ) {
            $class = ' wbls-notice-internal-page';
        }
        ?>
        <div class="notice notice-info is-dismissible wbls-notice-container<?php echo esc_attr($class); ?>">
            <div class="wbls-notice-col1"></div>
            <div class="wbls-notice-col2">
                <h3 class="wbls-notice-title"><?php esc_html_e('Thanks for using Whistleblowing System!','whistleblowing-system'); ?> 🤝</h3>
                <p class="wbls-notice-description">
                <?php
                esc_html_e('If you’re finding it useful, we’d really appreciate your review on WordPress.org — it helps us grow and keep improving.','whistleblowing-system');
                ?>

                </p>
                <div class="wbls-notice-button-row">
                    <?php
                    $review_redirect = 'https://wordpress.org/support/plugin/whistleblowing-system/reviews/#new-post';
                    $review_url      = wp_nonce_url(
                            add_query_arg(['wbls_rate_action' => 'review', 'go' => rawurlencode($review_redirect)]),
                            'wbls_rate_action'
                    );
                    ?>
                    <a href="<?php echo esc_url($review_url); ?>"
                       class="button wbls-notice-button"
                       target="_blank">
                        <?php esc_html_e('Leave a Review','whistleblowing-system'); ?>
                    </a>
                    <a href="<?php echo esc_url($remind_url); ?>"><?php esc_html_e('Remind Me Later','whistleblowing-system'); ?></a>
                    <a href="<?php echo esc_url($dismiss_url); ?>"><?php esc_html_e('Don’t Show Again','whistleblowing-system'); ?></a>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Handle clicks on the action links
     */
    public function handle_actions() {
        if ( empty($_GET['wbls_rate_action']) ) return;
        if ( ! isset($_GET['_wpnonce']) || ! wp_verify_nonce( sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'wbls_rate_action') ) return;

        $action = sanitize_text_field(wp_unslash($_GET['wbls_rate_action']));

        switch ( $action ) {
            case 'review':
                update_option(self::DISMISSED_KEY, 1); // hide permanently
                $target = isset($_GET['go']) ? esc_url_raw(wp_unslash($_GET['go'])) : 'https://wordpress.org/';
                wp_redirect($target);
                exit;

            case 'dismiss':
                update_option(self::DISMISSED_KEY, 1);
                break;

            case 'remind':
                update_option(self::REMIND_KEY, time() + DAY_IN_SECONDS * self::REMIND_AFTER_DAYS);
                break;
        }

        wp_safe_redirect(remove_query_arg(['wbls_rate_action', '_wpnonce']));
        exit;
    }
}

new WBLS_Rate_Notice();
