<?php
namespace WBLS_WhistleBlower\Free;

if ( ! defined('ABSPATH') ) exit;

class WhistleblowerLogs {

    private $table;
    private $per_page = 50;
    private $wpdb;

    public function __construct( $render_page = true ) {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to access this page.', 'whistleblowing-system'));
        }

        global $wpdb;
        $this->wpdb  = $wpdb;
        $this->table = $wpdb->prefix . 'wbls_logs';

        if ( $render_page ) {
            $this->handle_actions();
            $this->render_page();
        }
    }

    /**
     * Handle all actions (delete, export)
     */
    private function handle_actions() {
        // Handle bulk delete
        if ( isset($_POST['wbls_delete_selected']) ) {
            check_admin_referer('wbls_delete_logs', 'wbls_nonce');
            if ( ! current_user_can('manage_options') ) {
                wp_die( esc_html__('You do not have permission to delete logs.', 'whistleblowing-system') );
            }
            $this->handle_bulk_delete();
        }
    }

    /**
     * Bulk delete selected log records
     */
    private function handle_bulk_delete() {
        if ( !current_user_can('manage_options') ) {
            wp_die(esc_html__('You do not have permission to delete logs.', 'whistleblowing-system'));
        }

        check_admin_referer('wbls_delete_logs', 'wbls_nonce');

        $ids = isset($_POST['log_ids']) && is_array($_POST['log_ids']) ? array_map('intval', $_POST['log_ids']) : [];
        $ids = array_values(array_unique(array_filter($ids)));

        $deleted = 0;
        if ( $ids ) {
            foreach ( array_chunk($ids, 500) as $chunk ) {
                $placeholders = implode(',', array_fill(0, count($chunk), '%d'));
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                $sql = "DELETE FROM {$this->table} WHERE id IN ($placeholders)";

                // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
                $result = $this->wpdb->query( $this->wpdb->prepare($sql, $chunk) );
                if ( $result !== false ) {
                    $deleted += (int) $result;
                }
            }
        }

        if ($deleted > 0) {
            add_settings_error('wbls_logs', 'wbls_logs_deleted',
                    sprintf(_n('%d log deleted.', '%d logs deleted.', $deleted, 'whistleblowing-system'), $deleted),
                    'success'
            );
        }

        wp_safe_redirect( remove_query_arg(['wbls_delete_selected', 'log_ids']) );
        exit;
    }

    /**
     * Build WHERE SQL clause based on filters
     */
    private function build_where_clause($filters) {
        $where = "WHERE 1=1";

        // Define allowed filter keys to prevent SQL injection through column names
        $allowed_keys = ['form_id', 'submission_id', 'status', 'type', 'created_by'];

        foreach ($filters as $key => $value) {
            // Only process allowed keys and non-empty values
            if (in_array($key, $allowed_keys) && !empty($value)) {
                switch ($key) {
                    case 'form_id':
                    case 'submission_id':
                        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                        $where .= $this->wpdb->prepare(" AND {$key} = %d", (int)$value);
                        break;
                    case 'status':
                    case 'type':
                    case 'created_by':
                        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                        $where .= $this->wpdb->prepare(" AND {$key} = %s", sanitize_text_field($value));
                        break;
                }
            }
        }
        return $where;
    }

    /**
     * Fetch logs with filters and pagination
     */
    private function get_logs($filters, $order_by, $order, $offset, $per_page) {
        global $wpdb;

        $allowed_columns = [
                'id', 'created_at', 'form_id', 'submission_id',
                'type', 'status', 'message', 'created_by'
        ];

        // Validate inputs.
        $safe_order_by  = in_array($order_by, $allowed_columns, true) ? $order_by : 'created_at';
        $safe_order     = ($order === 'ASC') ? 'ASC' : 'DESC';
        $safe_offset    = absint($offset);
        $safe_per_page  = absint($per_page);

        // Build the WHERE clause (this method should return a fully sanitized string).
        $where = $this->build_where_clause($filters);

        // Build SQL directly inline (PHPCS prefers this over using a variable).
        $query = "
            SELECT *
            FROM {$this->table}
            {$where}
            ORDER BY {$safe_order_by} {$safe_order}
            LIMIT %d, %d
        ";


        return $wpdb->get_results(
                $wpdb->prepare(
                        $query, // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
                        $safe_offset,
                        $safe_per_page
                )
        );
    }

    /**
     * Get total number of items (for pagination)
     */
    private function get_total_items($filters) {
        $where = $this->build_where_clause($filters);

        // Table names cannot use placeholders in WordPress prepared statements
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $sql = "SELECT COUNT(*) FROM {$this->table} {$where}";

        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
        return (int) $this->wpdb->get_var($sql);
    }

    /**
     * Handle export to XLS
     */
    private function handle_export_xls() {}

    /**
     * Collect and sanitize filters
     */
    private function get_filters() {
        return [
                'form_id'    => isset($_GET['form_id']) ? intval($_GET['form_id']) : '',
                'submission_id'    => isset($_GET['submission_id']) ? intval($_GET['submission_id']) : '',
                'status'     => isset($_GET['status']) ? sanitize_text_field(wp_unslash($_GET['status'])) : '',
                'type'       => isset($_GET['type']) ? sanitize_text_field(wp_unslash($_GET['type'])) : '',
                'created_by' => isset($_GET['created_by']) ? sanitize_text_field(wp_unslash($_GET['created_by'])) : '',
        ];
    }

    /**
     * Render admin page (filters + table)
     */
    private function render_page() {
        $filters   = $this->get_filters();

        // Sanitize and unslash all GET parameters
        $order_by  = isset($_GET['orderby']) ? sanitize_text_field(wp_unslash($_GET['orderby'])) : 'created_at';
        $raw_order = isset($_GET['order']) ? sanitize_text_field(wp_unslash($_GET['order'])) : '';
        $order     = strtolower($raw_order) === 'asc' ? 'ASC' : 'DESC';
        $page      = isset($_GET['paged']) ? max(1, intval(wp_unslash($_GET['paged']))) : 1;
        $offset    = ($page - 1) * $this->per_page;

        $logs       = $this->get_logs($filters, $order_by, $order, $offset, $this->per_page);
        $total      = $this->get_total_items($filters);
        $total_pages = ceil($total / $this->per_page);

        // This query is correct - table names don't need placeholders
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        global $wpdb;

        $forms = $wpdb->get_results(
                $wpdb->prepare(
                        "SELECT ID, post_title FROM {$wpdb->posts} WHERE post_type = %s ORDER BY post_title ASC",
                        'wbls_form'
                )
        );
        $this->display($forms, $filters, $order_by, $order, $page, $logs, $total_pages);
    }

    private function display( $forms, $filters, $order_by, $order, $page, $logs, $total_pages ) {
        $raw      = get_option('wbls_global_settings', '{}');
        $settings = json_decode(is_string($raw) ? $raw : '{}', true) ?: [];
        $logs_active = isset($settings['logs_active']) ? (int) $settings['logs_active'] : 0;
        ob_start();
        if ( ! $logs_active ) {
            $this->render_deactive_log_screen();
        }
        WBLSLibrary::wbls_render_topbar_row(); ?>
        <div class="wrap wbls-logs-page">
            <h1><?php esc_html_e('Whistleblowing Logs', 'whistleblowing-system'); ?></h1>

            <form method="post" id="wbls-export-form">
                <?php wp_nonce_field('wbls_export', 'wbls_export_nonce'); ?>
                <input type="hidden" name="export_selected_ids" id="wbls-export-ids" value="">
            </form>
            <!-- 🔍 FILTERS FORM -->
            <form method="get" id="wbls-logs-filters">
                <input type="hidden" name="page" value="whistleblower_logs">

                <div class="filter-actions">
                    <a id="wbls-export-selected" class="button wbls-pro-tooltip wbls-pro-tooltip-action" href="<?php echo esc_url( wp_nonce_url( add_query_arg('export_xls', '1'), 'wbls_export' ) ); ?>">
                        📤 <?php esc_html_e('Export XLS', 'whistleblowing-system'); ?>
                    </a>
                </div>

                <div class="filters">
                    <div  class="filters-inputs">
                        <!-- Form -->
                        <div class="filter-field">
                            <label for="wbls_form_id"><?php esc_html_e('Form', 'whistleblowing-system'); ?></label>
                            <select name="form_id" id="wbls_form_id">
                                <option value=""><?php esc_html_e('All Forms', 'whistleblowing-system'); ?></option>
                                <?php foreach ($forms as $f): ?>
                                    <option value="<?php echo esc_attr($f->ID); ?>" <?php selected($filters['form_id'], $f->ID); ?>>
                                        <?php echo esc_html($f->post_title . ' ( #'.$f->ID.' )'); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>

                        <!-- Submission ID -->
                        <div class="filter-field" id="wbls-submission-field">
                            <label for="wbls_submission_id"><?php esc_html_e('Submission ID', 'whistleblowing-system'); ?></label>
                            <input type="number" name="submission_id" id="wbls_submission_id" value="<?php echo esc_attr($filters['submission_id']); ?>">
                        </div>

                        <!-- Status -->
                        <div class="filter-field">
                            <label><?php esc_html_e('Status', 'whistleblowing-system'); ?></label>
                            <select name="status">
                                <option value=""><?php esc_html_e('All', 'whistleblowing-system'); ?></option>
                                <option value="success" <?php selected($filters['status'], 'success'); ?>><?php esc_html_e('Success', 'whistleblowing-system'); ?></option>
                                <option value="error" <?php selected($filters['status'], 'error'); ?>><?php esc_html_e('Error', 'whistleblowing-system'); ?></option>
                            </select>
                        </div>

                        <!-- Type -->
                        <div class="filter-field">
                            <label><?php esc_html_e('Type', 'whistleblowing-system'); ?></label>
                            <select name="type">
                                <option value=""><?php esc_html_e('All', 'whistleblowing-system'); ?></option>
                                <option value="new_case" <?php selected($filters['type'], 'new_case'); ?>><?php esc_html_e('New Case', 'whistleblowing-system'); ?></option>
                                <option value="case_reply" <?php selected($filters['type'], 'case_reply'); ?>><?php esc_html_e('Case Reply', 'whistleblowing-system'); ?></option>
                                <option value="status_changed" <?php selected($filters['type'], 'status_changed'); ?>><?php esc_html_e('Status Changed', 'whistleblowing-system'); ?></option>
                                <option value="form_deleted" <?php selected($filters['type'], 'form_deleted'); ?>><?php esc_html_e('Form Deleted', 'whistleblowing-system'); ?></option>
                                <option value="submission_deleted" <?php selected($filters['type'], 'submission_deleted'); ?>><?php esc_html_e('Submission Deleted', 'whistleblowing-system'); ?></option>
                                <option value="outgoing_webhook" <?php selected($filters['type'], 'outgoing_webhook'); ?>><?php esc_html_e('Outgoing Webhook', 'whistleblowing-system'); ?></option>
                                <option value="incoming_webhook" <?php selected($filters['type'], 'incoming_webhook'); ?>><?php esc_html_e('Incoming Webhook', 'whistleblowing-system'); ?></option>
                            </select>
                        </div>

                        <?php $users = get_users(); ?>
                        <!-- Created By -->
                        <div class="filter-field">
                            <label><?php esc_html_e('Created by', 'whistleblowing-system'); ?></label>
                            <select name="created_by">
                                <option value=""><?php esc_html_e('All', 'whistleblowing-system'); ?></option>
                                <option value="user_na" <?php selected($filters['created_by'], 'user_na'); ?>><?php esc_html_e('Anonymous (user)', 'whistleblowing-system'); ?></option>
                                <option value="admin_na" <?php selected($filters['created_by'], 'admin_na'); ?>><?php esc_html_e('Anonymous (admin)', 'whistleblowing-system'); ?></option>
                                <?php
                                foreach ( $users as $user ) {
                                    ?>
                                    <option value="<?php echo esc_attr($user->user_login); ?>" <?php selected($filters['created_by'], esc_attr($user->user_login)); ?>><?php echo esc_html($user->user_login); ?></option>
                                    <?php
                                }
                                ?>
                            </select>
                        </div>
                    </div>

                    <div class="filter-actions">
                        <button type="submit" class="button button-primary">
                            <?php esc_html_e('Filter', 'whistleblowing-system'); ?>
                        </button>

                        <a class="button" href="<?php echo esc_url( admin_url('admin.php?page=whistleblower_logs') ); ?>">
                            ♻️ <?php esc_html_e('Reset Filters', 'whistleblowing-system'); ?>
                        </a>
                    </div>

                </div>

            </form>

            <?php settings_errors('wbls_logs'); ?>

            <!-- 🗑️ BULK DELETE FORM -->
            <form method="post" id="wbls-logs-bulk-form">
                <?php wp_nonce_field('wbls_delete_logs', 'wbls_nonce'); ?>

                <div class="wbls-bulk-actions">
                    <button type="submit" class="button button-secondary" name="wbls_delete_selected" id="wbls-delete-selected" disabled>
                        🗑️ <?php esc_html_e('Delete selected', 'whistleblowing-system'); ?>
                    </button>
                </div>

                <!-- 📜 TABLE -->
                <table class="widefat fixed striped wbls-logs-table" id="wbls-logs-table">
                    <thead>
                    <tr>
                        <th class="check-col"><input type="checkbox" id="wbls-check-all" /></th>
                        <?php
                        $columns = [
                                'id'             => __('ID', 'whistleblowing-system'),
                                'created_at'     => __('Date', 'whistleblowing-system'),
                                'form_id'        => __('Form', 'whistleblowing-system'),
                                'submission_id'  => __('Submission', 'whistleblowing-system'),
                                'type'           => __('Type', 'whistleblowing-system'),
                                'status'         => __('Status', 'whistleblowing-system'),
                                'message'        => __('Message', 'whistleblowing-system'),
                                'created_by'     => __('Created By', 'whistleblowing-system'),
                        ];
                        foreach ($columns as $col_key => $col_label):
                            $new_order = ($order_by === $col_key && $order === 'ASC') ? 'desc' : 'asc';
                            $arrow = $order_by === $col_key ? ($order === 'ASC' ? '▲' : '▼') : '';
                            $sort_url = add_query_arg(['orderby' => $col_key, 'order' => $new_order]);
                            ?>
                            <th><a href="<?php echo esc_url($sort_url); ?>"><?php echo esc_html($col_label . ' ' . $arrow); ?></a></th>
                        <?php endforeach; ?>
                    </tr>
                    </thead>

                    <tbody>
                    <?php if ($logs): foreach ($logs as $log):
                        $created_by = ($log->created_by === 'user_na') ? 'Anonymous (user)' : (($log->created_by === 'admin_na') ? 'Anonymous (admin)' : $log->created_by);
                        ?>
                        <tr class="status-<?php echo esc_attr($log->status); ?>">
                            <td class="check-col">
                                <input type="checkbox" class="wbls-row-check" name="log_ids[]" value="<?php echo (int)$log->id; ?>" />
                            </td>
                            <td><?php echo (int)$log->id; ?></td>
                            <td><?php echo esc_html($log->created_at); ?></td>
                            <td><?php echo $log->form_id ? (int)$log->form_id : '-'; ?></td>
                            <td><?php echo $log->submission_id ? (int)$log->submission_id : '-'; ?></td>
                            <td><?php echo esc_html($log->type); ?></td>
                            <td><?php echo esc_html($log->status); ?></td>
                            <td class="message">
                                <div class="short"><?php echo esc_html(wp_trim_words($log->message, 15)); ?></div>
                                <?php if (strlen($log->message) > 120): ?>
                                    <div class="full hidden"><?php echo esc_html($log->message); ?></div>
                                    <a href="#" class="toggle-msg"><?php esc_html_e('View more', 'whistleblowing-system'); ?></a>
                                <?php endif; ?>
                            </td>
                            <td><?php echo esc_html($created_by); ?></td>
                        </tr>
                    <?php endforeach; else: ?>
                        <tr><td colspan="10"><?php esc_html_e('No logs found.', 'whistleblowing-system'); ?></td></tr>
                    <?php endif; ?>
                    </tbody>
                </table>
            </form>

            <!-- 📄 PAGINATION -->
            <?php if ($total_pages > 1): ?>
                <div class="tablenav-pages wbls-pagination">
                    <?php $base_url = remove_query_arg('paged'); ?>
                    <?php if ($page > 1): ?>
                        <a class="button" href="<?php echo esc_url(add_query_arg('paged', $page - 1, $base_url)); ?>">&laquo; <?php esc_html_e('Previous', 'whistleblowing-system'); ?></a>
                    <?php endif; ?>

                    <span class="current-page">
                    <?php echo esc_html(sprintf(__('Page %d of %d', 'whistleblowing-system'), $page, $total_pages)); ?>
                </span>

                    <?php if ($page < $total_pages): ?>
                        <a class="button" href="<?php echo esc_url(add_query_arg('paged', $page + 1, $base_url)); ?>"><?php esc_html_e('Next', 'whistleblowing-system'); ?> &raquo;</a>
                    <?php endif; ?>
                </div>
            <?php endif; ?>
        </div>
        <?php
        echo wp_kses(ob_get_clean(), WBLSLibrary::$wp_kses_form);
    }

    private function render_deactive_log_screen() {
        ?>
        <div class="wbls-layer"></div>
        <div class="wbls-logs-inactive-popup">
            <h2><?php esc_html_e('Enable logs to view activity', 'whistleblowing-system'); ?></h2>
            <p><?php esc_html_e('Logging is currently disabled. Turn on Logs in the settings to start capturing events and populate this page.', 'whistleblowing-system'); ?></p>
            <a href="?page=whistleblower_settings#wbls-tab-logs"><?php esc_html_e('Enable logs', 'whistleblowing-system'); ?> >></a>
        </div>
        <?php
    }
}

