<?php
namespace WBLS_WhistleBlower\Free;
use WP_Query;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

class WhistleblowerSubmissionEdit {

    public $fields = [];
    public $form_id;
    public $submissions = [];
    public $settings = [];
    public $global_settings = [];
    public $whistleblower_active = false;
    public $anonymous = false;
    public $anonymous_active = false;

    public function __construct() {
        $task = isset($_GET['task']) ? sanitize_text_field(wp_unslash($_GET['task'])) : '';
        $this->form_id = isset($_GET['id']) ? intval($_GET['id']) : 0;
        $this->get_submissions();
        $this->get_form_fields();
        $this->whistleblower_active = WBLSLibrary::is_whistleblower_active( $this->form_id );
        $this->settings = get_post_meta($this->form_id, 'wbls_form_settings', 1);
        $this->global_settings = json_decode( get_option( 'wbls_global_settings' ), 1 );
        if ( method_exists($this, $task) ) {
            $this->$task();
        } else {
            $this->display();
        }
    }

    public function get_submissions() {
        $orderby = isset($_GET['orderby']) ? sanitize_key($_GET['orderby']) : 'date';
        $order   = isset($_GET['order']) ? sanitize_text_field(wp_unslash($_GET['order'])) : 'DESC';
        $order   = (strtoupper($order) === 'ASC') ? 'ASC' : 'DESC';

        // If sorting by status, first ensure all posts have the meta with default 0.
        if ($orderby === 'status') {
            $ids = get_posts([
                    'post_type'      => 'wbls_form_subm',
                    'post_status'    => 'closed',
                    'fields'         => 'ids',
                    'nopaging'       => true,
                    'no_found_rows'  => true,
                    'meta_query'     => [
                            [
                                    'key'     => 'wbls_form_id',
                                    'value'   => $this->form_id,
                                    'compare' => '=',
                            ],
                    ],
            ]);
            foreach ($ids as $pid) {
                if ( ! metadata_exists('post', $pid, 'wbls_submission_status') ) {
                    add_post_meta($pid, 'wbls_submission_status', 0, true);
                }
            }
        }

        // Map UI orderby
        $meta_key    = null;
        $orderby_arg = null;
        switch ($orderby) {
            case 'status':
                $meta_key    = 'wbls_submission_status';
                $orderby_arg = 'meta_value_num';
                break;
            case 'date':
            default:
                $meta_key    = 'wbls_created_at';   // unix timestamp
                $orderby_arg = 'meta_value_num';
                break;
        }

        $args = [
                'post_type'      => 'wbls_form_subm',
                'post_status'    => 'closed',
                'meta_query'     => [
                        [
                                'key'     => 'wbls_form_id',
                                'value'   => $this->form_id,
                                'compare' => '=',
                        ],
                ],
                'meta_key'       => $meta_key,
                'orderby'        => $orderby_arg,
                'order'          => $order,
                'posts_per_page' => 1000,
                'no_found_rows'  => true,
        ];

        $query = new \WP_Query($args);
        $this->submissions = array_map(fn($p) => $p->ID, $query->posts ?? []);
    }

    public function get_form_fields() {
        $fields = get_post_meta($this->form_id, 'wbls_field_options', true);
        $this->fields = $fields;
    }

    public function display() {
        $file_max_size = isset($this->settings['file_max_size']) ? floatval($this->settings['file_max_size']) : 10;
        $ajaxnonce = wp_create_nonce('wbls_ajax_nonce');
        $export_url = add_query_arg(array(
            'form_id' => $this->form_id,
            'nonce' => $ajaxnonce,
            'action' => 'wbls_admin_ajax',
            'task' => 'download_csv_file'
        ), admin_url('admin-ajax.php'));

        wp_enqueue_style(WBLS_PREFIX . '-submissions');
        wp_enqueue_style(WBLS_PREFIX . '-style');
        wp_enqueue_script( WBLS_PREFIX . '-submissions');
        wp_localize_script(WBLS_PREFIX . '-submissions', 'wbls_submissions', array(
            "ajaxnonce" => $ajaxnonce,
            'file_size_msg' =>  esc_html__("File size should be less then", 'whistleblowing-system'),
            'file_type_msg' =>  esc_html__("Invalid file type: allowed types are", 'whistleblowing-system'),
            'file_max_size' =>  $file_max_size,
            'file_types' =>  $this->settings['file_types'] ?? ['jpg', 'png', 'gif'],
            'export_url' => $export_url,
            'submission_success_delete' => esc_html__("Submission successfully deleted", 'whistleblowing-system'),
            'submission_error_delete' => esc_html__("Something went wrong", 'whistleblowing-system'),
        ));


        $current_orderby = isset($_GET['orderby']) ? sanitize_key($_GET['orderby']) : 'date';
        $current_order   = isset($_GET['order']) && strtoupper(sanitize_text_field(wp_unslash($_GET['order']))) === 'ASC' ? 'ASC' : 'DESC';
        WBLSLibrary::wbls_render_topbar_row(); ?>
        <div class="wbls-admin-header">
            <img class="wbls-admin-header-logo" src="<?php echo esc_url(WBLS_URL . '/admin/assets/images/whistleblowing_logo.png'); ?>">
            <h2 class="wbls-page-title">
                <?php esc_html_e('Submissions of', 'whistleblowing-system'); ?>
                <?php echo esc_html(get_the_title($this->form_id)); ?>
                <?php esc_html_e('form', 'whistleblowing-system'); ?>
            </h2>
            <span class="wbls-button wbls-export-csv wbls-pro-tooltip-action"><?php esc_html_e('Export CSV', 'whistleblowing-system') ?></span>
        </div>
        <p class="wbls-response-message wbls-hidden"></p>
        <div class="wbls-bulk-action-row">
            <select class="wbls-bulk-actions">
                <option value="-1"><?php esc_html_e('Bulk Actions', 'whistleblowing-system'); ?></option>
                <option value="delete"><?php esc_html_e('Delete', 'whistleblowing-system'); ?></option>
                <option value="activate"><?php esc_html_e('Activate', 'whistleblowing-system'); ?></option>
                <option value="block"><?php esc_html_e('Block', 'whistleblowing-system'); ?></option>
                <option value="complete"><?php esc_html_e('Complete', 'whistleblowing-system'); ?></option>
            </select>
            <span class="button wbls-bulk-action-apply"><?php esc_html_e('Apply', 'whistleblowing-system'); ?></span>
        </div>
        <div class="wrap wbls-content">
            <table class="wp-list-table wbls-subm-table">
                <thead>
                <tr>
                    <th><input type="checkbox" name="all" data-id="all" class="wbls-all-submissions"></th>
                    <th>ID</th>
                    <?php
                    foreach ( $this->fields as $field ) {
                        if( empty($field) || $field['type'] == 'submit' || $field['type'] == 'recaptcha' || $field['type'] == 'page_break' ) {
                            continue;
                        } elseif ( $field['type'] == 'checkbox' ) {
                            foreach ( $field['options'] as $option ) {
                                $shortText = strip_tags($option['miniLabel']);
                                if(strlen($shortText) > 20) {
                                    $shortText = substr($shortText, 0, 20) . '...';
                                }
                                ?>
                                <th title="<?php echo esc_attr($option['miniLabel']); ?>"><?php echo esc_html(wp_strip_all_tags($shortText)); ?></th>
                                <?php
                            }
                        }
                        else {
                            $shortText = $field['label'];
                            if( empty($field['label']) ) {
                                $shortText = ucfirst($field['type']) .' '. esc_html__('field', 'whistleblowing-system');
                            }
                            elseif(strlen($field['label']) > 20) {
                                $shortText = substr($shortText, 0, 20) . '...';
                            }
                            ?>
                            <th title="<?php echo esc_attr($field['label']); ?>"><?php echo esc_html(wp_strip_all_tags($shortText)); ?></th>
                            <?php
                        }
                    }
                    ?>
                    <th>
                        <a href="<?php echo esc_url($this->wbls_sort_url('date', $current_orderby, $current_order)); ?>" class="wbls-sort-link">
                            <?php esc_html_e('Date', 'whistleblowing-system'); ?>
                            <?php echo esc_html( $this->wbls_sort_indicator('date', $current_orderby, $current_order) ); ?>
                        </a>
                    </th>
                    <?php if ( $this->whistleblower_active || $this->anonymous_active ) { ?>
                        <th><?php esc_html_e('Chat', 'whistleblowing-system') ?></th>
                        <th><?php esc_html_e('Access', 'whistleblowing-system') ?></th>
                        <th>
                            <a href="<?php echo esc_url($this->wbls_sort_url('status', $current_orderby, $current_order)); ?>" class="wbls-sort-link">
                                <?php esc_html_e('Status', 'whistleblowing-system'); ?>
                                <?php echo esc_html( $this->wbls_sort_indicator('status', $current_orderby, $current_order) ); ?>
                            </a>
                        </th>
                    <?php } ?>
                </tr>
                </thead>
                <tbody>
                <?php
                $count = count($this->submissions);
                foreach ( $this->submissions as $submission_id ) { ?>
                    <tr>
                        <td><input type="checkbox" data-id="<?php echo intval($submission_id) ?>" class="wbls-single-submissions"></td>
                        <td><?php echo intval($submission_id) ?>
                            <div class="wbls-row-actions row-actions">
                          <span class="wbls-delete-submission" data-submissionId="<?php echo intval($submission_id); ?>" data-formId="<?php echo intval($this->form_id); ?>">
                              <?php esc_html_e('Delete', 'whistleblowing-system'); ?>
                          </span>
                                <span class="wbls-edit-submission" data-submissionId="<?php echo intval($submission_id); ?>" data-formId="<?php echo intval($this->form_id); ?>">
                              <a href="?page=whistleblower_submission_item_edit&form_id=<?php echo intval($this->form_id); ?>&submission_id=<?php echo intval($submission_id); ?>"><?php esc_html_e('Edit', 'whistleblowing-system'); ?></a>
                          </span>
                            </div>

                        </td>
                        <?php
                        foreach ( $this->fields as $field ) {
                            if( empty($field) || $field['type'] == 'submit' || $field['type'] == 'recaptcha' || $field['type'] == 'page_break') continue;

                            $field_value = '';
                            if($field['name'] !== '') {
                                $field_value = WBLS_Encryption::decrypt(get_post_meta($submission_id, $field['name'], true));
                            }

                            if( $field['type'] == 'file' && !empty($field_value)) {


                                // First, decrypt if it's encrypted (both old and new submissions)
                                $decrypted_value = WBLS_Encryption::decrypt($field_value);
                                
                                // Check if it's an array (new encrypted file info) or string (old format)
                                if (is_array($decrypted_value)) {
                                    // NEW FORMAT: Array containing file info
                                    $file_infos = isset($decrypted_value[0]) ? $decrypted_value : [$decrypted_value];
                                    ?>
                                    <td>
                                        <?php foreach ($file_infos as $index => $file_info):
                                            if (!is_array($file_info)) continue;

                                            $is_encrypted = isset($file_info['encrypted']) && $file_info['encrypted'];
                                            $original_name = isset($file_info['original_name']) ? $file_info['original_name'] : 'file';
                                            $file_path = isset($file_info['path']) ? $file_info['path'] : '';
                                            $file_size = isset($file_info['file_size']) ? $file_info['file_size'] : 0;

                                            // Get file extension
                                            $ext = strtolower(pathinfo($original_name, PATHINFO_EXTENSION));
                                            $is_image = in_array($ext, ['jpg', 'jpeg', 'png', 'gif', 'bmp', 'webp']);

                                            // Generate unique ID for modal
                                            $file_id = 'file_' . $submission_id . '_' . sanitize_title($field['name']) . '_' . $index;

                                            if ($is_encrypted) {
                                                // ENCRYPTED FILE: Compact icon-based UI
                                                ?>
                                                <div class="wbls-file-item">
                                                    <div class="wbls-file-item-container">
                                                        <!-- File icon with status -->
                                                        <div class="wbls-file-icon-col">
                                                            <?php if ($is_image): ?>
                                                                <div class="wbls-file-icon wbls-file-icon-image wbls-show-encrypted-image"
                                                                     data-attr-file_id="<?php echo $file_id; ?>"
                                                                     title="Encrypted Image - Click to preview">
                                                                    <span class="dashicons dashicons-format-image"></span>
                                                                </div>
                                                            <?php else: ?>
                                                                <div class="wbls-file-icon" title="Encrypted File">
                                                                    <span class="dashicons dashicons-media-default"></span>
                                                                </div>
                                                            <?php endif; ?>
                                                        </div>

                                                        <!-- File info (hidden in tooltip) -->
                                                        <div class="wbls-file-info-col">
                                                            <div class="wbls-file-name-row">
                                                                <?php echo esc_html($this->wbls_truncate_filename($original_name, 20)); ?>
                                                            </div>
                                                            <div  class="wbls-file-meta-row">
                                                                <?php echo strtoupper($ext); ?> ·
                                                                <?php echo $file_size ? size_format($file_size, 1) : 'Unknown size'; ?>
                                                                <span><span class="dashicons dashicons-lock"></span> Encrypted</span>
                                                            </div>
                                                        </div>

                                                        <!-- Action icons -->
                                                        <div  class="wbls-file-action-col">
                                                            <?php if ($is_image): ?>
                                                                <button type="button"
                                                                        class="button wbls-icon-button wbls-show-encrypted-image"
                                                                        data-attr-file_id="<?php echo $file_id; ?>"
                                                                        title="Preview image">
                                                                    <span class="dashicons dashicons-visibility"></span>
                                                                </button>
                                                            <?php endif; ?>

                                                            <button
                                                               class="button button-primary wbls-icon-button wbls-download-action-btn"
                                                               data-attr-file_id="<?php echo $file_id; ?>"
                                                               title="Download file">
                                                                <span class="dashicons dashicons-download"></span>
                                                            </button>
                                                        </div>
                                                    </div>

                                                    <!-- Hidden modal trigger -->
                                                    <div id="<?php echo $file_id; ?>" style="display: none;"
                                                         data-submission-id="<?php echo $submission_id; ?>"
                                                         data-field-name="<?php echo esc_attr($field['name']); ?>"
                                                         data-file-index="<?php echo $index; ?>"
                                                         data-file-name="<?php echo esc_attr($original_name); ?>"
                                                         data-is-encrypted="1">
                                                    </div>
                                                </div>
                                                <?php
                                            } else {
                                                // NON-ENCRYPTED FILE: Compact icon-based UI
                                                if ( $file_path && file_exists(WBLS_UPLOAD_DIR . '/' . $file_path) ) {
                                                    $file_url = WBLS_UPLOAD_URL . '/' . basename($file_path);
                                                    ?>
                                                    <div class="wbls-file-item">
                                                        <div class="wbls-file-item-container">
                                                            <!-- File icon -->
                                                            <div class="wbls-file-icon-col">
                                                                <?php if($is_image): ?>
                                                                    <div class="wbls-file-icon wbls-file-icon-image wbls-show-image"
                                                                         data-file-url="<?php echo esc_url($file_url); ?>"
                                                                         data-file-item="<?php echo esc_attr($original_name); ?>"
                                                                         title="Image - Click to preview">
                                                                        <img src="<?php echo esc_url($file_url); ?>">
                                                                    </div>
                                                                <?php elseif($ext == 'pdf'): ?>
                                                                    <div class="wbls-file-icon"
                                                                         title="PDF Document">
                                                                        <span class="dashicons dashicons-pdf"></span>
                                                                    </div>
                                                                <?php elseif($ext == 'mp3' || $ext == 'wav'): ?>
                                                                    <div class="wbls-file-icon"
                                                                         title="Audio File">
                                                                        <span class="dashicons dashicons-media-audio"></span>
                                                                    </div>
                                                                <?php else: ?>
                                                                    <div class="wbls-file-icon"
                                                                         title="File">
                                                                        <span class="dashicons dashicons-media-default" style="font-size: 20px; color: #6b7280;"></span>
                                                                    </div>
                                                                <?php endif; ?>
                                                            </div>

                                                            <!-- File info (hidden in tooltip) -->
                                                            <div class="wbls-file-info-col">
                                                                <div class="wbls-file-name-row">
                                                                    <?php echo esc_html($this->wbls_truncate_filename($original_name, 20)); ?>
                                                                </div>
                                                                <div class="wbls-file-meta-row">
                                                                    <?php echo strtoupper($ext); ?>
                                                                    <?php if ($file_size): ?>
                                                                        · <?php echo size_format($file_size, 1); ?>
                                                                    <?php endif; ?>
                                                                </div>
                                                            </div>

                                                            <!-- Action icons -->
                                                            <div class="wbls-file-action-col">
                                                                <?php if ($is_image): ?>
                                                                    <button type="button"
                                                                            class="button wbls-icon-button wbls-show-image"
                                                                            data-file-url="<?php echo esc_url($file_url); ?>"
                                                                            data-file-item="<?php echo esc_attr($original_name); ?>"
                                                                            title="Preview image">
                                                                        <span class="dashicons dashicons-visibility"></span>
                                                                    </button>
                                                                <?php endif; ?>

                                                                <a href="<?php echo esc_url($file_url); ?>"
                                                                   target="_blank"
                                                                   class="button button-primary wbls-icon-button"
                                                                   title="Download file">
                                                                    <span class="dashicons dashicons-download"></span>
                                                                </a>
                                                            </div>
                                                        </div>
                                                    </div>
                                                    <?php
                                                }
                                            }
                                        endforeach; ?>
                                    </td>
                                    <?php
                                } else {
                                    // OLD FORMAT: Comma-separated file paths (backward compatibility)
                                    $field_values = explode(',', $decrypted_value ?: $field_value);
                                    ?>
                                    <td>
                                        <?php foreach ($field_values as $file_item):
                                            if (empty($file_item)) continue;

                                            $file_url = WBLS_UPLOAD_URL . '/' . $file_item;
                                            $ext = strtolower(pathinfo($file_item, PATHINFO_EXTENSION));
                                            $is_image = in_array($ext, ['jpg', 'jpeg', 'png', 'gif']);
                                            ?>
                                            <div class="wbls-file-item">
                                                <div class="wbls-file-item-container">
                                                    <!-- File icon -->
                                                    <div class="wbls-file-icon-col">
                                                        <?php if ( $is_image ): ?>
                                                            <div class="wbls-file-icon wbls-show-image"
                                                                 data-file-url="<?php echo esc_url($file_url); ?>"
                                                                 data-file-item="<?php echo esc_attr($file_item); ?>"
                                                                 title="Image - Click to preview">
                                                                <img src="<?php echo esc_url($file_url); ?>">
                                                            </div>
                                                        <?php elseif($ext == 'pdf'): ?>
                                                            <div class="wbls-file-icon"
                                                                 title="PDF Document">
                                                                <span class="dashicons dashicons-pdf"></span>
                                                            </div>
                                                        <?php else: ?>
                                                            <div class="wbls-file-icon"
                                                                 title="File">
                                                                <span class="dashicons dashicons-media-default"></span>
                                                            </div>
                                                        <?php endif; ?>
                                                    </div>

                                                    <!-- File info -->
                                                    <div class="wbls-file-info-col">
                                                        <div class="wbls-file-name-row">
                                                            <?php echo esc_html($this->wbls_truncate_filename($file_item, 20)); ?>
                                                        </div>
                                                        <div class="wbls-file-meta-row">
                                                            <?php echo strtoupper($ext); ?>
                                                        </div>
                                                    </div>

                                                    <!-- Action icons -->
                                                    <div class="wbls-file-action-col">
                                                        <?php if ($is_image): ?>
                                                            <button type="button"
                                                                    class="button wbls-icon-button wbls-show-image"
                                                                    data-file-url="<?php echo esc_url($file_url); ?>"
                                                                    data-file-item="<?php echo esc_attr($file_item); ?>"
                                                                    title="Preview image">
                                                                <span class="dashicons dashicons-visibility"></span>
                                                            </button>
                                                        <?php endif; ?>

                                                        <a href="<?php echo esc_url($file_url); ?>"
                                                           target="_blank"
                                                           class="button button-primary wbls-icon-button"
                                                           title="Download file">
                                                            <span class="dashicons dashicons-download"></span>
                                                        </a>
                                                    </div>
                                                </div>
                                            </div>
                                        <?php endforeach; ?>
                                    </td>
                                    <?php
                                }
                            }
                            elseif ( $field['type'] == 'fullName' ) {
                                $firstName = $field_value['firstName'] ?? '';
                                $middleName = $field_value['middleName'] ?? '';
                                $lastName = $field_value['lastName'] ?? '';
                                $fullName = $firstName . ' ' . $middleName . ' ' . $lastName;
                                ?>
                                <td><?php echo esc_html($fullName); ?></td>
                                <?php
                            }
                            elseif ( $field['type'] == 'address' ) {
                                $addressJoined = '';
                                if( is_array($field_value) ) {
                                    foreach ($field_value as $a) {
                                        if ($a != '') {
                                            $addressJoined .= $a . ', ';
                                        }
                                    }
                                    $addressJoined = substr($addressJoined, 0, -2);
                                }
                                ?>
                                <td><?php echo esc_html($addressJoined); ?></td>
                                <?php
                            }
                            elseif ( $field['type'] == 'checkbox' ) {
                                foreach ( $field['options'] as $option ) {
                                    $field_value =  WBLS_Encryption::decrypt(get_post_meta($submission_id, $option['name'], true));

                                    if( $field_value ) {
                                        $field_value = 'Checked';
                                    }
                                    ?>
                                    <td><?php echo esc_html($field_value); ?></td>
                                    <?php
                                }
                            }
                            elseif ( $field['type'] == 'textarea' ) {
                                $shortText = $field_value;
                                if(strlen($field_value) > 50) $shortText = substr($shortText, 0, 50).'...';
                                ?>
                                <td class="wbls-textarea" title="<?php echo esc_attr($field_value); ?>"><?php echo esc_html($shortText); ?></td>
                                <?php
                            }
                            elseif ( $field['type'] == 'anonymous' ) {
                                if( intval($field_value) == 1 ) {
                                    $this->anonymous = true;
                                    $field_value = 'Checked';
                                } else {
                                    $this->anonymous = false;
                                    $field_value = 'Unchecked';
                                }
                                ?>
                                <td><?php echo esc_html($field_value); ?></td>
                                <?php
                            }
                            elseif ( $field['type'] == 'DateTime' ) {
                                $date = $field_value['date'] ?? '';
                                $time = $field_value['time'] ?? '';
                                $day = $field_value['day'] ?? '';
                                $month = $field_value['month'] ?? '';
                                $year = $field_value['year'] ?? '';
                                $dateTime = $time;
                                if( $date ) {
                                    $dateTime = $date.' '.$time;
                                } elseif( $day || $month || $year ) {
                                    $dateTime = esc_html($day).'/'.esc_html($month).'/'.esc_html($year).' '.esc_html($time);
                                }
                                ?>
                                <td><?php echo esc_html($dateTime); ?></td>
                                <?php
                            }
                            else {
                                if( gettype($field_value) !== 'array' ) {
                                    ?>
                                    <td><?php echo esc_html($field_value); ?></td>
                                    <?php
                                } else { ?>
                                    <td><?php echo esc_html(json_encode($field_value)); ?></td>
                                    <?php
                                }
                            }
                        } ?>
                        <td>
                            <?php
                            $created_at = get_post_meta($submission_id, 'wbls_created_at', true);
                            echo esc_html(date("Y-m-d H:i:s", $created_at));
                            ?>
                        </td>
                        <?php
                        if ( $this->whistleblower_active || $this->anonymous) {
                            $admin_token = WBLS_Encryption::decrypt(get_post_meta($submission_id, 'wbls_admin_token', true));
                            $user_token = WBLS_Encryption::decrypt(get_post_meta($submission_id, 'wbls_user_token', true));
                            $status_id = get_post_meta( $submission_id, 'wbls_submission_status', 1 );
                            if( $status_id === false || $status_id === '' ) {
                                $status_id = 0;
                            }
                            $statuses = ['Active', 'Completed', 'Blocked'];
                            ?>
                            <td class="wbls-access-chat-column">
                                <span class="wbls-chat-icon" title="<?php esc_attr_e('Open Chat', 'whistleblowing-system'); ?>"></span>
                                <?php $this->chat($submission_id); ?>
                            </td>
                            <td class="wbls-access-key-column">
                                <span class="wbls-access-icon" title="<?php esc_attr_e('Get access tokens', 'whistleblowing-system') ?>"></span>
                                <div class="wbls-access-key-container">
                                    <div class="wbls-access-key-item wbls-access-key-admin">
                                        <label><?php esc_html_e('Admin login token', 'whistleblowing-system') ?></label>
                                        <div class="wbls-token-wrapper">
                                            <input type="password" readonly value="<?php echo esc_attr($admin_token); ?>" class="wbls-token-input" />
                                            <span class="wbls-token-toggle" title="Show/Hide"></span>
                                            <span class="wbls-token-copy" title="Copy"></span>
                                        </div>
                                    </div>
                                    <?php

                                    if( isset($this->global_settings['user_token_visibility_active']) && $this->global_settings['user_token_visibility_active'] ) { ?>
                                        <div class="wbls-access-key-item wbls-access-key-user">
                                            <label><?php esc_html_e('User login token', 'whistleblowing-system') ?></label>
                                            <div class="wbls-token-wrapper">
                                                <input type="password" readonly value="<?php echo esc_attr($user_token); ?>" class="wbls-token-input" />
                                                <span class="wbls-token-toggle" title="Show/Hide"></span>
                                                <span class="wbls-token-copy" title="Copy"></span>
                                            </div>
                                        </div>
                                    <?php } else { ?>
                                        <a class="wbls-access-key-user-enable-link" href="?page=whistleblower_settings#wbls-tab-advanced"><?php esc_html_e('Want to see the user token? Enable it from the settings page.', 'whistleblowing-system') ?></a>
                                    <?php } ?>
                                </div>
                            </td>
                            <td class="wbls-status-column">
                                <spam class="wbls-status-button" title="Click to edit">
                                    <span data-status="<?php echo intval($status_id); ?>" data-submission_id=<?php echo intval($submission_id); ?> data-form_id=<?php echo intval($this->form_id); ?>  class="wbls-status-button-title"><?php echo esc_html($statuses[$status_id]); ?></span>
                                    <div class="wbls-hidden wbls-status-dropdown">
                                        <?php foreach ($statuses as $key => $status ) { ?>
                                            <span data-status="<?php echo intval($key); ?>" class="wbls-status-item">
                                            <?php echo esc_html($status); ?>
                                        </span>
                                        <?php } ?>
                                    </div>
                                </spam>
                            </td>
                        <?php } elseif ($this->anonymous_active) { ?>
                            <td class="wbls-access-chat-column">
                            </td>
                            <td class="wbls-access-key-column">
                            </td>
                            <td class="wbls-status-column">
                            </td>
                        <?php } ?>
                    </tr>
                <?php } ?>
                </tbody>
            </table>
        </div>
        <?php
    }

    // Helper function to truncate long filenames
    function wbls_truncate_filename($filename, $length = 20) {
        if (strlen($filename) <= $length) return $filename;
        $ext = pathinfo($filename, PATHINFO_EXTENSION);
        $name = pathinfo($filename, PATHINFO_FILENAME);
        $truncated = substr($name, 0, $length - 3 - strlen($ext)) . '...';
        return $truncated . ($ext ? '.' . $ext : '');
    }


    public function wbls_sort_url($orderby, $current_orderby, $current_order) {
        $next = 'ASC';
        if ($orderby === $current_orderby) {
            $next = ($current_order === 'ASC') ? 'DESC' : 'ASC';
        }
        return esc_url( add_query_arg(array(
                'page'    => isset($_GET['page']) ? sanitize_key($_GET['page']) : '',
                'id'      => isset($_GET['id']) ? intval($_GET['id']) : '',
                'orderby' => $orderby,
                'order'   => $next,
        )) );
    }

    public function wbls_sort_indicator($orderby, $current_orderby, $current_order) {
        if ($orderby !== $current_orderby) return '';
        return $current_order === 'ASC' ? ' ▲' : ' ▼';
    }

    public function chat( $submission_id ) {
        $chats = get_post_meta( $submission_id, 'wbls_chat', 1 );
        if( !$chats ) $chats = [];
        $admin_token = WBLS_Encryption::decrypt(get_post_meta( $submission_id, 'wbls_admin_token', 1 ));
        ?>
        <div class="wbls-chats-content" style="display:none">

            <div class="wbls-chat-container">
                <div class="wbls-submission-chat-header">
                    <div class="wbls-submission-chat-header-title">Messages</div>
                    <div class="wbls-chats-close"></div>
                </div>
                <div class="wbls-chats-section">
                    <?php
                    foreach ($chats as $chat ) {
                        $message = WBLS_Encryption::decrypt($chat['message']);
                        $files = [];
                        $file_data = WBLS_Encryption::decrypt($chat['file']);

                        // Handle file data (could be string or array)
                        if ( is_array($file_data) && isset($file_data[0]['path']) ) {
                            // NEW FORMAT: Array containing file info
                            if (isset($file_data[0]) && is_array($file_data[0])) {
                                // Multiple files
                                $files = $file_data;
                            } else {
                                // Single file
                                $files = [$file_data];
                            }
                        } elseif ( ! empty($file_data) ) {
                            // OLD FORMAT: Comma-separated string

                            if( !is_array($file_data) ) {
                                $tempFiles[] = $file_data;
                            } else {
                                $tempFiles = $file_data;
                            }
                            foreach ($tempFiles as $file_path) {
                                $files[] = [
                                        'path' => $file_path,
                                        'encrypted' => false,
                                        'original_name' => basename($file_path),
                                        'is_old_format' => true
                                ];
                            }
                        }
                        ?>
                        <div class="wbls_<?php echo esc_html($chat['role']); ?>_row">
                            <div class="wbls_message_col">
                                <span class="wbls_message_role">
                                    <?php echo esc_html($chat['role'])." / "; ?>
                                    <?php echo esc_html( date('d-m-Y H:i:s', $chat['modified_date']) ); ?>
                                </span>
                                <?php if( $message != '' ) { ?>
                                    <span class="wbls_message"><?php echo esc_html($message); ?></span>
                                <?php } ?>
                                <?php
                                foreach ( $files as $index => $file_info ) {
                                    if (empty($file_info)) continue;

                                    // Extract file information
                                    $is_encrypted = isset($file_info['encrypted']) && $file_info['encrypted'];
                                    $original_name = isset($file_info['original_name']) ? $file_info['original_name'] : 'file';
                                    $file_path = isset($file_info['path']) ? $file_info['path'] : '';
                                    $file_size = isset($file_info['file_size']) ? $file_info['file_size'] : 0;

                                    // Get file extension
                                    $ext = strtolower(pathinfo($original_name, PATHINFO_EXTENSION));
                                    $is_image = in_array($ext, ['jpg', 'jpeg', 'png', 'gif', 'bmp', 'webp']);

                                    // Generate unique ID
                                    $file_id = 'chat_file_' . $submission_id . '_' . $chat['modified_date'] . '_' . $index;

                                    if ($is_encrypted) {
                                        // ENCRYPTED FILE in chat
                                        ?>
                                        <div class="wbls-chat-file-item">
                                            <div class="wbls-chat-file-container">
                                                <!-- File icon -->
                                                <div class="wbls-chat-file-icon">
                                                    <?php if ($is_image): ?>
                                                        <div class="wbls-chat-file-image wbls-show-encrypted-image"
                                                             data-attr-file_id="<?php echo $file_id; ?>"
                                                             title="Encrypted Image - Click to preview">
                                                            <span class="dashicons dashicons-format-image"></span>
                                                            <span class="wbls-encrypted-badge">
                                                            <span class="dashicons dashicons-lock"></span>
                                                        </span>
                                                        </div>
                                                    <?php else: ?>
                                                        <div class="wbls-chat-file-icon-default" title="Encrypted File">
                                                            <span class="dashicons dashicons-media-default"></span>
                                                            <span class="wbls-encrypted-badge">
                                                            <span class="dashicons dashicons-lock"></span>
                                                        </span>
                                                        </div>
                                                    <?php endif; ?>
                                                </div>

                                                <!-- File info -->
                                                <div class="wbls-chat-file-info">
                                                    <div class="wbls-chat-file-name">
                                                        <?php echo esc_html($this->wbls_truncate_filename($original_name, 25)); ?>
                                                    </div>
                                                    <div class="wbls-chat-file-meta">
                                                        <?php echo strtoupper($ext); ?>
                                                        <?php if ($file_size): ?>
                                                            · <?php echo size_format($file_size, 1); ?>
                                                        <?php endif; ?>
                                                    </div>

                                                    <!-- Action buttons -->
                                                    <div class="wbls-chat-file-actions">
                                                        <?php if ($is_image): ?>
                                                            <button type="button"
                                                                    class="wbls-chat-action-btn wbls-show-encrypted-image"
                                                                    data-attr-file_id="<?php echo $file_id; ?>"
                                                                    title="Preview">
                                                                <span class="dashicons dashicons-visibility"></span>
                                                            </button>
                                                        <?php endif; ?>

                                                        <button type="button"
                                                                class="wbls-chat-action-btn wbls-download-action-btn"
                                                                data-attr-file_id="<?php echo $file_id; ?>"
                                                                title="Download">
                                                            <span class="dashicons dashicons-download"></span>
                                                        </button>
                                                    </div>
                                                </div>
                                            </div>

                                            <!-- Hidden data for encrypted files -->
                                            <div id="<?php echo $file_id; ?>" style="display: none;"
                                                 data-submission-id="<?php echo $submission_id; ?>"
                                                 data-field-name="wbls_chat"
                                                 data-file-index="<?php echo $index; ?>"
                                                 data-chat-timestamp="<?php echo $chat['modified_date']; ?>"
                                                 data-file-name="<?php echo esc_attr($original_name); ?>"
                                                 data-is-encrypted="1">
                                            </div>
                                        </div>
                                        <?php
                                    } else {
                                        // NON-ENCRYPTED FILE in chat
                                        $file_url = WBLS_UPLOAD_URL . '/' . basename($file_path);
                                        ?>
                                        <div class="wbls-chat-file-item">
                                            <div class="wbls-chat-file-container">
                                                <!-- File icon/thumbnail -->
                                                <div class="wbls-chat-file-icon">
                                                    <?php if ($is_image): ?>
                                                        <div class="wbls-chat-file-image wbls-show-image"
                                                             data-file-url="<?php echo esc_url($file_url); ?>"
                                                             data-file-name="<?php echo esc_attr($original_name); ?>"
                                                             title="Click to preview">
                                                            <img src="<?php echo esc_url($file_url); ?>" alt="<?php echo esc_attr($original_name); ?>">
                                                        </div>
                                                    <?php elseif($ext == 'pdf'): ?>
                                                        <div class="wbls-chat-file-icon-pdf" title="PDF Document">
                                                            <span class="dashicons dashicons-pdf"></span>
                                                        </div>
                                                    <?php elseif($ext == 'mp3' || $ext == 'wav'): ?>
                                                        <div class="wbls-chat-file-icon-audio" title="Audio File">
                                                            <span class="dashicons dashicons-media-audio"></span>
                                                        </div>
                                                    <?php else: ?>
                                                        <div class="wbls-chat-file-icon-default" title="File">
                                                            <span class="dashicons dashicons-media-default"></span>
                                                        </div>
                                                    <?php endif; ?>
                                                </div>

                                                <!-- File info -->
                                                <div class="wbls-chat-file-info">
                                                    <div class="wbls-chat-file-name">
                                                        <a href="<?php echo esc_url($file_url); ?>"
                                                           target="_blank"
                                                           title="<?php echo esc_attr($original_name); ?>">
                                                            <?php echo esc_html($this->wbls_truncate_filename($original_name, 25)); ?>
                                                        </a>
                                                    </div>
                                                    <div class="wbls-chat-file-meta">
                                                        <?php echo strtoupper($ext); ?>
                                                        <?php if ($file_size): ?>
                                                            · <?php echo size_format($file_size, 1); ?>
                                                        <?php endif; ?>
                                                    </div>

                                                    <!-- Action buttons -->
                                                    <div class="wbls-chat-file-actions">
                                                        <?php if ($is_image): ?>
                                                            <button type="button"
                                                                    class="wbls-chat-action-btn wbls-show-image"
                                                                    data-file-url="<?php echo esc_url($file_url); ?>"
                                                                    data-file-name="<?php echo esc_attr($original_name); ?>"
                                                                    title="Preview">
                                                                <span class="dashicons dashicons-visibility"></span>
                                                            </button>
                                                        <?php endif; ?>

                                                        <a href="<?php echo esc_url($file_url); ?>"
                                                           target="_blank"
                                                           class="wbls-chat-action-btn"
                                                           title="Download">
                                                            <span class="dashicons dashicons-download"></span>
                                                        </a>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        <?php
                                    }
                                } ?>
                            </div>
                        </div>
                    <?php } ?>
                </div>
                <div class="wbls-new-chat-section">
                    <form class="wbls-reply-form" id="wbls-reply-form_<?php echo intval($submission_id); ?>">
                        <?php if( WBLS_PRO ) { ?>
                            <input type="hidden" name="action" value="wbls_admin_ajax">
                            <input type="hidden" name="task" value="wbls_reply">
                            <input type="hidden" value="<?php echo intval($this->form_id); ?>" name="wbls_form_id">
                            <input type="hidden" value="<?php echo esc_attr($admin_token); ?>" name="wbls-admin-token" class="wbls-admin-token">
                            <input type="hidden" value="<?php echo intval($submission_id); ?>" name="wbls-ticket_id" class="wbls-ticket_id">
                        <?php } ?>
                        <textarea name="reply" class="wbls-new-reply" placeholder="Type here..."></textarea>
                        <div class="wbls-reply-button-container">
                            <!--PRO start-->
                            <div class="wbls-reply-attachement-cont">
                                <span class="imageName"></span>
                                <label for="wbls-file-input_<?php echo intval($submission_id); ?>">
                                    <img title="Attachment" src="<?php echo esc_url(WBLS_URL . '/frontend/assets/images/upload.svg'); ?>"/>
                                </label>
                                <input id="wbls-file-input_<?php echo intval($submission_id); ?>" type="file" name="wbls-attachement[]" multiple="multiple" class="wbls-reply-attachement wbls-file-input" accept="image/*,.pdf,audio/*,video/*">
                            </div>
                            <!--PRO end-->
                            <button class="wbls-reply-button<?php echo !WBLS_PRO ? ' wbls-pro-tooltip-action' : '';?>"><?php esc_html_e('Send', 'whistleblowing-system') ?></button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
        <?php
    }

    /* Pro start */
    public function display_file_preview() {
        
    }
}
