(function () {
    const { registerBlockType } = wp.blocks;
    const { __ } = wp.i18n;
    const { InspectorControls, useBlockProps } = wp.blockEditor || wp.editor;
    const { PanelBody, SelectControl, ToggleControl, TextControl, Spinner, Notice } = wp.components;
    const { useSelect } = wp.data;
    const ServerSideRender = wp.serverSideRender;
    const { useState, Fragment, useEffect } = wp.element;

    const icon =
        (window.WBLS_BLOCK && WBLS_BLOCK.iconUrl)
            ? wp.element.createElement('img', {
                src: WBLS_BLOCK.iconUrl,
                width: 24,
                height: 24,
                alt: '',
                style: { display: 'block' }
            })
            : 'feedback'; // fallback

    registerBlockType('wbls/form', {
        icon,
        edit: function (props) {
            const { attributes, setAttributes } = props;
            const { formId, displayType, themeId } = attributes;
            const blockProps = useBlockProps();

            // Try to fetch forms if the CPT is exposed to REST.
            const formsQuery = { per_page: -1, _fields: 'id,title' };
            const [forms, setForms] = useState([]);
            const [themes, setThemes] = useState([]);
            const [isLoading, setIsLoading] = useState(true);



            // Direct fetch on component mount
            useEffect(() => {
                async function fetchForms() {
                    try {
                        setIsLoading(true);

                        // Direct REST API call
                        const response = await wp.apiFetch({
                            path: '/wp/v2/wbls_form?per_page=-1&_fields=id,title'
                        });

                        setForms(response || []);
                    } catch (error) {
                        console.error('Failed to fetch forms:', error);
                        setForms([]);
                    } finally {
                        setIsLoading(false);
                    }
                }
                fetchForms();

                async function fetchThemes() {
                    try {
                        setIsLoading(true);

                        // Direct REST API call
                        const response = await wp.apiFetch({
                            path: '/wp/v2/wbls_theme?per_page=-1&_fields=id,title'
                        });

                        setThemes(response || []);
                    } catch (error) {
                        console.error('Failed to fetch forms:', error);
                        setForms([]);
                    } finally {
                        setIsLoading(false);
                    }
                }
                fetchThemes();
            }, []);

            // In your block's edit function
            useEffect(() => {
                if (themeId > 0) {
                    // Remove any existing theme CSS
                    const existingLinks = document.querySelectorAll('link[data-wbls-theme]');
                    existingLinks.forEach(link => link.remove());

                    // Add theme CSS
                    const themeCssUrl = `${window.WBLS_BLOCK.themeBaseUrl}/wbls-system/wbls-theme-style_${themeId}.css?t=${Date.now()}`;
                    const link = document.createElement('link');
                    link.rel = 'stylesheet';
                    link.href = themeCssUrl;
                    link.setAttribute('data-wbls-theme', themeId);
                    document.head.appendChild(link);

                } else {
                    // Remove theme CSS when theme is 0
                    const themeLinks = document.querySelectorAll('link[data-wbls-theme]');
                    themeLinks.forEach(link => link.remove());
                }

                // Cleanup
                return () => {
                    const themeLinks = document.querySelectorAll('link[data-wbls-theme]');
                    themeLinks.forEach(link => link.remove());
                };
            }, [themeId]);

            useEffect(() => {
                const SELECTOR = '.wp-block-wbls-form .wblsform-page-and-images';
                const INTERVAL_MS = 200;
                const MAX_ATTEMPTS = 50; // 50 * 200ms = 10000ms = 10s

                // Remove any existing active classes
                document.querySelectorAll('.wblsform-active-page')
                    .forEach(el => el.classList.remove('wblsform-active-page'));

                function activateFirstPage() {
                    const pageElements = document.querySelectorAll(SELECTOR);
                    if (pageElements.length > 0) {
                        pageElements[0].classList.add('wblsform-active-page');
                        return true;
                    }
                    return false;
                }

                // Fast-path: try immediately
                if (activateFirstPage()) return;

                let attempts = 0;
                const intervalId = setInterval(() => {
                    attempts += 1;

                    if (activateFirstPage() || attempts >= MAX_ATTEMPTS) {
                        clearInterval(intervalId);
                    }
                }, INTERVAL_MS);

                return () => {
                    clearInterval(intervalId);
                };
            }, [formId, displayType, themeId]);


            // Create form options
            const formOptions = [{
                label: __('— Select a form —', 'whistleblowing-system'),
                value: 0
            }];

            if (Array.isArray(forms)) {
                forms.forEach((form) => {
                    const title = form.title?.rendered || form.title || 'Untitled';
                    formOptions.push({
                        label: `#${form.id} — ${title.replace(/<[^>]*>/g, '')}`,
                        value: form.id
                    });
                });
            }

            // Create form options
            const themeOptions = [{
                label: __('— Select a theme —', 'whistleblowing-system'),
                value: 0
            }];

            if (Array.isArray(themes)) {
                themes.forEach((theme) => {
                    const title = theme.title?.rendered || theme.title || 'Untitled';
                    themeOptions.push({
                        label: `#${theme.id} — ${title.replace(/<[^>]*>/g, '')}`,
                        value: theme.id
                    });
                });
            }

            // Display type options
            const displayTypeOptions = [
                { label: __('Default (Tabbed Interface)', 'whistleblowing-system'), value: 'tab' },
                { label: __('Buttons View (Popup Interface)', 'whistleblowing-system'), value: 'popup' },
                { label: __('Form Only', 'whistleblowing-system'), value: 'form' },
                { label: __('Login Screen', 'whistleblowing-system'), value: 'login' }
            ];

            return wp.element.createElement(
                wp.element.Fragment,
                {},
                wp.element.createElement(
                    InspectorControls,
                    {},
                    // SECTION 1: Form Selection
                    wp.element.createElement(
                        PanelBody,
                        {
                            title: __('Form Selection', 'whistleblowing-system'),
                            initialOpen: true
                        },
                        isLoading
                            ? wp.element.createElement(
                                'div',
                                { style: { textAlign: 'center' } },
                                wp.element.createElement(Spinner, null),
                                wp.element.createElement('p', null, __('Loading forms...', 'whistleblowing-system'))
                            )
                            : forms === undefined || forms === null
                                ? wp.element.createElement(
                                    'div',
                                    { className: 'components-notice is-error' },
                                    wp.element.createElement('p', null,
                                        __('Error loading forms. Please check console.', 'whistleblowing-system')
                                    )
                                )
                                : wp.element.createElement(SelectControl, {
                                    label: __('Choose Form', 'whistleblowing-system'),
                                    value: formId || 0,
                                    options: formOptions,
                                    onChange: (val) => setAttributes({
                                        formId: parseInt(val, 10) || 0
                                    }),
                                })
                    ),

                    // SECTION 2: Display Settings (only shown when form is selected)
                    formId > 0 && wp.element.createElement(
                        PanelBody,
                        {
                            title: __('Display Settings', 'whistleblowing-system'),
                            initialOpen: true
                        },
                        wp.element.createElement(SelectControl, {
                            label: __('Display Type', 'whistleblowing-system'),
                            value: displayType || 'tab',
                            options: displayTypeOptions,
                            onChange: (val) => setAttributes({
                                displayType: val || 'tab'
                            }),
                            help: __('Choose how to display the form', 'whistleblowing-system')
                        })
                    ),
                    formId > 0 && wp.element.createElement(
                        PanelBody,
                        {
                            title: __('Theme Selection', 'whistleblowing-system'),
                            initialOpen: true
                        },
                        wp.element.createElement(
                            wp.element.Fragment,
                            {},
                            wp.element.createElement(SelectControl, {
                                label: __('Choose Theme', 'whistleblowing-system'),
                                value: themeId || 0,
                                options: themeOptions,
                                onChange: (val) => setAttributes({
                                    themeId: parseInt(val, 10) || 0
                                }),
                            }),
                        )
                    ),
                ),

                // Main block content
                !formId || formId === 0
                    ? wp.element.createElement(
                        'div',
                        { ...blockProps, className: (blockProps.className || '') + ' wbls-form-placeholder' },
                        wp.element.createElement(Notice, { status: 'info', isDismissible: false },
                            __('Select a form from the sidebar settings.', 'whistleblowing-system')
                        )
                    )
                    : wp.element.createElement(
                        'div',
                        blockProps,
                        wp.element.createElement(ServerSideRender, {
                            block: 'wbls/form',
                            attributes: attributes
                        })
                    )
            );
        },
        save: function () { return null; } // dynamic block
    });
})();
