<?php
namespace WBLS_WhistleBlower\Free;
use WP_Query;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly
class WBLSFront_Controller {
    public $submissions = array();
    public $form_data = array();
    public $form_settings = array();
    public $form_id = 0;
    public $submission_id = 0;
    public $form_title = '';
    public $user_token = '';
    public $admin_token = '';

    public $anonymous = false;

    public function __construct( $task ) {
        if ( method_exists($this, $task) && $task != '__construct' ) {
                $this->$task();
        }
    }

    public function wbls_reply() {
        $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
        if ( ! wp_verify_nonce( $nonce, 'wbls_ajax_nonce' ) ) {
            die( esc_html__( 'Security check', 'whistleblowing-system' ) );
        }
        $wbls_security = isset($_POST['wbls_security']) ? sanitize_text_field(wp_unslash($_POST['wbls_security'])) : '';
        if( $wbls_security != "" ) {
            wp_send_json_error(array(
                'message' => esc_html__('Security issue!', 'whistleblowing-system'),
            ));
        }

        $reply = isset( $_POST['reply'] ) ? sanitize_text_field(wp_unslash($_POST['reply'])) : '';
        $token = isset( $_POST['token'] ) ? sanitize_text_field(wp_unslash($_POST['token'])) : '';
        $form_id = isset( $_POST['wbls_form_id'] ) ? intval($_POST['wbls_form_id']) : 0;
        if( $form_id == 0 ) {
            wp_send_json_error(array(
                'message' => esc_html__('Security issue!', 'whistleblowing-system'),
            ));
        }

        $file_path = "";
        $chat_attachments = [];

        if( ($reply == "" && empty($chat_attachments)) || $token == "" ) {
            wp_send_json_error(array(
                'message' => esc_html__('Something went wrong', 'whistleblowing-system'),
            ));
        }
        $encrypted = WBLS_Encryption::encrypt($token);
        $args = array(
            'post_type'		=>	'wbls_form_subm',
            'post_status' => 'closed',
            'meta_query'	=>	array(
                'relation' => 'OR',
                array(
                    'key'   => 'wbls_admin_token',
                    'value' => $token, // unencrypted value (legacy)
                ),
                array(
                    'key'   => 'wbls_admin_token',
                    'value' => $encrypted,
                ),
                array(
                    'key'   => 'wbls_user_token',
                    'value' => $token,
                ),
                array(
                    'key'   => 'wbls_user_token',
                    'value' => $encrypted,
                ),
            ),
            "numberposts" => 1000,
            "posts_per_page" => 1000,
        );
        $query = new WP_Query( $args );
        $post_id = 0;
        $admin_token_active = 1;
        if ( $query->posts ) {
            foreach ( $query->posts as $post ) {
                $post_id = $post->ID;
                $temp_token = get_post_meta($post_id, 'wbls_user_token', 1);
                $decrypt_temp_token = WBLS_Encryption::decrypt($temp_token);
                if( $token === $decrypt_temp_token ) {
                    $admin_token_active = 0;
                }
            }
        } else {
            \WBLS_WhistleBlower\Free\WBLS_Logger::log( 'login', 'error', 'Token is not correct', [
                'form_id' => $form_id,
                'created_by' => 'user_na',
            ]);
            wp_send_json_error(array(
                'message' => esc_html__('Token is not correct, please check and try one more time', 'whistleblowing-system'),
            ));
        }

        $role = 'user';
        if( $admin_token_active ) {
            $role = 'admin';
        }
        $modified_date = current_time( 'timestamp' );
        $chat = array(
            'message' => WBLS_Encryption::encrypt(sanitize_text_field($reply)),
            'file' => !empty($chat_attachments) ? WBLS_Encryption::encrypt($chat_attachments) : [],
            'published' => 1,
            'role' => $role,
            'modified_date' => $modified_date,
        );

        $chats = get_post_meta( $post_id, 'wbls_chat', 1 );

        $chats[] = $chat;

        $message_insert = update_post_meta($post_id, 'wbls_chat', $chats, false);

        if ( $message_insert ) {
            $last_chat = end($chats);

            // Prepare just this new message for response
            $new_message = array(
                'text' => WBLS_Encryption::decrypt($last_chat['message']),
                'attachment' => [],
                'role' => $last_chat['role'],
                'date' => gmdate("Y-m-d H:i:s", $modified_date),
                'timestamp' => $modified_date
            );

            $role = ($role === 'admin') ? 'admin_na' : 'user_na';
            \WBLS_WhistleBlower\Free\WBLS_Logger::log( 'case_reply', 'success', 'New Whistleblowing Form reply', [
                'form_id' => $form_id,
                'submission_id' => $post_id,
                'created_by' => sanitize_text_field($role),
            ]);

            wp_send_json_success(array(
                'new_message' => $new_message,
                'submission_id' => $post_id,
            ), 200 );

        }
        $role = ($role === 'admin') ? 'admin_na' : 'user_na';
        \WBLS_WhistleBlower\Free\WBLS_Logger::log( 'case_reply', 'error', 'Reply message not inserted to DB', [
            'form_id' => $form_id,
            'submission_id' => $post_id,
            'created_by' => sanitize_text_field($role),
        ]);


        wp_send_json_error(array(
            'message' => esc_html__('Something went wrong', 'whistleblowing-system'),
        ));
    }

    public function wbls_login() {
        $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
        if ( ! wp_verify_nonce( $nonce, 'wbls_ajax_nonce' ) ) {
            die( esc_html__( 'Security check', 'whistleblowing-system' ) );
        }
        $wbls_security = isset($_POST['wbls_security']) ? sanitize_text_field(wp_unslash($_POST['wbls_security'])) : '';
        if( $wbls_security != "" ) {
            wp_send_json_error(array(
                'message' => esc_html__('Security issue!', 'whistleblowing-system'),
            ));
        }
        $token = isset($_POST['wbls_token']) ? sanitize_text_field(wp_unslash($_POST['wbls_token'])) : '';
        if( $token == '' ) {
            wp_send_json_error(array(
                'message' => esc_html__('Token issue, please check and try one more time', 'whistleblowing-system'),
            ));
        }

        $encrypted = WBLS_Encryption::encrypt($token);
        $args = array(
            'post_type'	=>	'wbls_form_subm',
            'post_status' => 'closed',
            'meta_query'	=>	array(
                'relation' => 'OR',
                array(
                    'key'   => 'wbls_admin_token',
                    'value' => $token, // unencrypted value (legacy)
                ),
                array(
                    'key'   => 'wbls_admin_token',
                    'value' => $encrypted,
                ),
                array(
                    'key'   => 'wbls_user_token',
                    'value' => $token,
                ),
                array(
                    'key'   => 'wbls_user_token',
                    'value' => $encrypted,
                ),
            ),
            "numberposts" => 1000,
            "posts_per_page" => 1000,
        );
        $query = new WP_Query( $args );
        $post_id = 0;
        $admin_token_active = 1;
        $status_id = 0;
        if ( $query->posts ) {
            foreach ( $query->posts as $post ) {
                $post_id = $post->ID;
                $status_id = get_post_meta( $post_id, 'wbls_submission_status', 1 );
                if( intval($status_id) === 2 ) {
                    wp_send_json_error(array(
                        'message' => esc_html__('Your token is blocked', 'whistleblowing-system'),
                    ));
                }
                $temp_token = get_post_meta($post_id, 'wbls_user_token', 1);
                $decrypt_token = WBLS_Encryption::decrypt($token);
                if( $token === $decrypt_token ) {
                    $admin_token_active = 0;
                }
            }
        } else {
            wp_send_json_error(array(
                'message' => esc_html__('Token is not correct, please check and try one more time', 'whistleblowing-system'),
            ));
        }
        $chats = get_post_meta($post_id, 'wbls_chat', 1);
        if ( empty($chats) ) {
            wp_send_json_error(array(
                'message' => esc_html__('There is no any message in this ticket, please open new ticket', 'whistleblowing-system'),
            ));
        }

        $form_id = 0;
        if( $post_id ) {
            $form_id = get_post_meta( $post_id, 'wbls_form_id', true );
        }

        $chat_status = 'wbls-chat-status-active';
        $chat_status_text = esc_html__('Chat status is active', 'whistleblowing-system');
        if( intval($status_id) === 1 ) {
            $chat_status = 'wbls-chat-status-completed';
            $chat_status_text = esc_html__('Chat status is completed', 'whistleblowing-system');
        }

        $response_data = array(
            'admin_token_active' => $admin_token_active,
            'form_id' => $form_id,
            'message' => array(),
            'chat_status' => $chat_status,
            'chat_status_text' => $chat_status_text
        );
        foreach ( $chats as $chat ) {
            if( !empty($chat) ) {
                $files = [];
                $message = WBLS_Encryption::decrypt($chat['message']);
                $response_data['message'][] = [
                    'text' => $message,
                    'attachment' => $files,
                    'role' => $chat['role'],
                    'date' => gmdate("Y-m-d H:i:s", $chat['modified_date']),
                    'modified_date' => $chat['modified_date'],
                    'submission_id' => $post_id
                ];
            }
        }

        wp_send_json_success(array(
            'chats' => $response_data,
        ), 200 );

    }

    public function wbls_submit_form() {

        $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
        if ( ! wp_verify_nonce( $nonce, 'wbls_ajax_nonce' ) ) {
            wp_send_json_error(array(
                'message' => esc_html__("Security check", "whistleblowing-system"),
            ));
        }
        $form_id = isset($_POST['wbls_form_id']) ? intval($_POST['wbls_form_id']) : 0;
        $wbls_security = isset($_POST['wbls_security']) ? sanitize_text_field(wp_unslash($_POST['wbls_security'])) : '';
        if( !$form_id || $wbls_security != "" ) {
            wp_send_json_error(array(
                'message' => esc_html__("Security check", "whistleblowing-system"),
            ));
        }

        $this->form_data = get_post_meta( $form_id, 'wbls_field_options', true );
        $this->form_settings = get_post_meta( $form_id, 'wbls_form_settings', true );
        $this->form_title = get_the_title( $form_id );
        $this->form_id = $form_id;
        $submission = array();
        $required_validation = true;
        $file_path = '';
        $chatMessage = '';
        $wbls_hidden_conditions = isset($_POST['wbls_hidden_conditions']) ? sanitize_text_field(wp_unslash($_POST['wbls_hidden_conditions'])) : '';
        if( $wbls_hidden_conditions !== '' ) {
            $wbls_hidden_conditions = explode(",",$wbls_hidden_conditions);
            $wbls_hidden_conditions = array_map(function ($item) {
                return preg_replace('/_(l|m|f|street|street1|city|state|postal|country)$/', '', $item);
            }, $wbls_hidden_conditions);
        }

        foreach ( $this->form_data as $data ) {
            if( empty($data) || empty($data['name']) && $data['type'] !== 'checkbox' ) {
                continue;
            }

            $name = $data['name'];

            /* Check if field hidden by condition and skip required check */
            if( $wbls_hidden_conditions === '' || !in_array($name, $wbls_hidden_conditions) ) {
                /* Check required */
                if ($data['type'] == 'fullName') {
                    if ((empty($_POST[$name . '_f']) ||
                        (empty($_POST[$name . '_m']) && intval($data['hideMiddleName']) != 0) ||
                        empty($_POST[$name . '_l'])) && $data['required']) {
                        $required_validation = false;
                        break;
                    }
                } elseif ($data['type'] == 'address') {
                    if ((empty($_POST[$name . '_street']) ||
                        empty($_POST[$name . '_city']) ||
                        empty($_POST[$name . '_state']) ||
                        empty($_POST[$name . '_postal']) ||
                        empty($_POST[$name . '_country'])) && $data['required'] ) {
                        $required_validation = false;
                        break;
                    }
                } elseif ($data['type'] == 'DateTime') {
                    if ( ( (isset($_POST[$name . '_date']) && ($_POST[$name . '_date'] === '' || $_POST[$name . '_date'] === '0') ) ||
                            (isset($_POST[$name . '_time']) && ($_POST[$name . '_time'] === '' || $_POST[$name . '_time'] === '0') ) ||
                            (isset($_POST[$name . '_day']) && ($_POST[$name . '_day'] === '' || $_POST[$name . '_day'] === '0') ) ||
                            (isset($_POST[$name . '_month']) && ($_POST[$name . '_month'] === '' || $_POST[$name . '_month'] === '0') ) ||
                            (isset($_POST[$name . '_year']) && ($_POST[$name . '_year'] === '' || $_POST[$name . '_year'] === '0'))) && $data['required'] ) {
                        $required_validation = false;
                        break;
                    }
                } elseif ($data['type'] == 'checkbox') {
                    if( $data['required_single'] === 1 ) {
                        $temp_req = false;
                        foreach ($data['options'] as $option) {
                            $name = $option['name'];
                            if (!empty($_POST[$name]) && $data['required']) {
                                $temp_req = true;
                                break;
                            }
                        }
                        $required_validation = $temp_req;
                    } else {
                        foreach ($data['options'] as $option) {
                            $name = $option['name'];
                            if (empty($_POST[$name]) && $data['required']) {
                                $required_validation = false;
                                break;
                            }
                        }
                    }
                } else {
                    if (empty($_POST[$name]) && isset($data['required']) && $data['required'] && !isset($_FILES[$name])) {
                        $required_validation = false;
                        break;
                    }
                }
            }

            if ( $data['type'] == 'fullName' ) {
                $fName = isset($_POST[$name.'_f']) ? sanitize_text_field(wp_unslash($_POST[$name.'_f'])) : '';
                $mName = isset($_POST[$name.'_m']) ? sanitize_text_field(wp_unslash($_POST[$name.'_m'])) : '';
                $lName = isset($_POST[$name.'_l']) ? sanitize_text_field(wp_unslash($_POST[$name.'_l'])) : '';
                $submission[$name] = array(
                    'firstName' => $fName,
                    'middleName' => $mName,
                    'lastName' => $lName,
                );
            }
            elseif ( $data['type'] == 'address' ) {
                $street = isset($_POST[$name.'_street']) ? sanitize_text_field(wp_unslash($_POST[$name.'_street'])) : '';
                $street1 = isset($_POST[$name.'_street1']) ? sanitize_text_field(wp_unslash($_POST[$name.'_street1'])) : '';
                $city = isset($_POST[$name.'_city']) ? sanitize_text_field(wp_unslash($_POST[$name.'_city'])) : '';
                $state = isset($_POST[$name.'_state']) ? sanitize_text_field(wp_unslash($_POST[$name.'_state'])) : '';
                $postal = isset($_POST[$name.'_postal']) ? sanitize_text_field(wp_unslash($_POST[$name.'_postal'])) : '';
                $country = isset($_POST[$name.'_country']) ? sanitize_text_field(wp_unslash($_POST[$name.'_country'])) : '';
                $submission[$name] = array(
                    'street' => $street,
                    'street1' => $street1,
                    'city' => $city,
                    'state' => $state,
                    'postal' => $postal,
                    'country' => $country,
                );
            } elseif( $data['type'] == 'checkbox' ) {
                foreach ( $data['options'] as $option ) {
                    $name = $option['name'];
                    $submission[$name] = isset($_POST[$name]) ? sanitize_text_field(wp_unslash($_POST[$name])) : '';
                }
            }
            elseif ( isset($_POST[$name]) ) {
                $submission[$name] = isset($_POST[$name]) ? sanitize_text_field(wp_unslash($_POST[$name])) : '';
                if ($data['type'] == 'textarea') {
                    $chatMessage = $submission[$name];
                }
            }
        }
        $this->submissions = $submission;
        if( !$required_validation ) {
            wp_send_json_error();
        }

        $my_post = array(
            'post_status'   => 'closed',
            'post_author'   => 1,
            'post_type' => 'wbls_form_subm'
        );

        /* Create submission post */
        $insert = wp_insert_post($my_post);
        if( $insert ) {
            $this->submission_id = $insert;
            foreach ( $submission as $key => $subm ) {
                $encrypted = WBLS_Encryption::encrypt($subm);
                add_post_meta( $insert, $key, $encrypted, true );
            }
            add_post_meta( $insert, 'wbls_created_at', current_time( 'timestamp' ), true );
            add_post_meta( $insert, 'wbls_form_id', intval($form_id), true );

            $whistleblower_active = WBLSLibrary::is_whistleblower_active($form_id);
            if( $whistleblower_active || $this->anonymous) {
                $wbls_global_settings = json_decode( get_option( 'wbls_global_settings' ), true );
                $token_length = 32;
                if( WBLS_PRO ) {
                    $token_length = isset($wbls_global_settings['token_length']) ? intval($wbls_global_settings['token_length']) : 32;
                }
                $this->user_token = $this->random_base64url($token_length);
                $this->admin_token = $this->random_base64url($token_length);

                $encrypted_admin_token = WBLS_Encryption::encrypt($this->admin_token);
                $encrypted_user_token = WBLS_Encryption::encrypt($this->user_token);

                add_post_meta( $insert, 'wbls_admin_token', $encrypted_admin_token, true );
                add_post_meta( $insert, 'wbls_user_token', $encrypted_user_token, true );


                $chat[] = array(
                    'message' => WBLS_Encryption::encrypt(sanitize_text_field($chatMessage)),
                    'file' => !empty($chat_attachments) ? WBLS_Encryption::encrypt($chat_attachments) : '',
                    'published' => 1,
                    'role' => 'user',
                    'modified_date' => current_time( 'timestamp' ),
                );
                $message_insert = add_post_meta( $insert, 'wbls_chat', $chat, true );
                if ( $message_insert ) {
                    $this->wbls_send_email( $form_id );
                    \WBLS_WhistleBlower\Free\WBLS_Logger::log( 'new_case', 'success', 'New Whistleblowing Form submission', [
                        'form_id' => $form_id,
                        'submission_id' => $this->submission_id,
                        'created_by' => 'user_na',
                    ]);
                    wp_send_json_success( array(
                        'whistleblower_active' => $whistleblower_active,
                        'token' => $this->user_token,
                        'anonymous' => $this->anonymous,
                    ), 200 );
                }
            } else {
                $this->wbls_send_email( $form_id );
                \WBLS_WhistleBlower\Free\WBLS_Logger::log( 'new_case', 'success', 'New Form submission', [
                    'form_id' => $form_id,
                    'submission_id' => $this->submission_id,
                    'created_by' => 'user_na',
                ]);
                wp_send_json_success( array(
                    'whistleblower_active' => $whistleblower_active
                ), 200 );
            }

        }
        wp_send_json_error();
    }

    private function random_base64url($len = 6) {
        // each 4 base64 chars encode 3 bytes -> need ceil($len/4)*3 bytes
        $bytes = random_bytes((int)ceil($len * 3 / 4));
        $b64   = rtrim(strtr(base64_encode($bytes), '+/', '-_'), '=');
        return substr($b64, 0, $len);
    }


    private function wbls_send_email( $form_id, $subject = '', $body = '' ) {

        $settings = get_post_meta( $form_id, 'wbls_email_options', true );
        $subjectTemp = $subject;
        if( isset($settings['sendemail']) && $settings['sendemail'] && isset($settings['admin_mail']) && $settings['admin_mail'] != '') {
            $to = esc_html($settings['admin_mail']);
            if( $subject == '' ) {
                $subject = !empty($settings['mail_subject']) ? esc_html($this->convert_body_placeholders($settings['mail_subject'], 1, 1)) : 'Whistleblower new message';
            }
            $from_name = !empty($settings['from_name']) ? esc_html($settings['from_name']) : 'Whistleblower';
            $from_mail = !empty($settings['wbls_mail_from']) ? esc_html($settings['wbls_mail_from']) : '';
            if( $body == '' ) {
                $body = !empty($settings['wbls_mail_body']) ? wp_kses($this->convert_body_placeholders($settings['wbls_mail_body'], 0, 1), WBLSLibrary::$wp_kses_default) : 'You have new message from the Whistleblower form';
            }
            $headers[] = "Content-type: text/html; charset=UTF-8";
            $headers[] = "MIME-Version: 1.0";
            if ( $from_mail != '' ) {
                $headers[] = "From: " . $from_name . " <" . $from_mail . ">";
            } else {
                $headers[] = "From: " . $from_name . " <" . get_bloginfo('admin_email') . ">";
            }
            wp_mail($to, $subject, $body, $headers);
        }
    }

    private function get_email_fields() {
        $emails = [];
        foreach ($this->form_data as $data ) {
            if ( empty($data) || $data['type'] != 'email' ) {
                continue;
            }
            $emails[] = $this->submissions[$data['name']];
        }
        return $emails;
    }

    private function convert_body_placeholders( $body, $is_subject, $admin = 1 ) {
        $replacements = [
            '{Form title}'      => 'Form: ' . $this->form_title,
            '{Form ID}'         => 'Form ID: ' . $this->form_id,
            '{Submission ID}'   => 'Submission ID: ' . $this->submission_id,
            '{Admin Token}'   => "Admin Token: " . $this->admin_token,
            '{User Token}'   => "User Token: " . $this->user_token,
        ];
        $body = strtr($body, $replacements);

        $all_data = '';
        foreach ( $replacements as $key => $val ) {
            if( ($admin && $key === '{User Token}') || (!$admin && $key !== '{User Token}') ) {
                continue;
            }
            $all_data .= "<p>".$val."</p>";
        }

        // Reorder form fields as in html view using order setting in the field data
        if (!empty($this->form_data)) {
            // Sort array by 'order' value ascending
            usort($this->form_data, function($a, $b) {
                // If 'order' is missing, treat as 0
                $orderA = isset($a['order']) ? (int)$a['order'] : 0;
                $orderB = isset($b['order']) ? (int)$b['order'] : 0;

                return $orderA <=> $orderB;
            });
        }


        foreach ($this->form_data as $data ) {
            if ( empty($data) ||
                ($data['type'] === 'page_break' || $data['type'] === 'submit' || $data['type'] === 'file' ||
                (!isset($this->submissions[$data['name']]) && $data['type'] != 'checkbox' )) ) {
                continue;
            }

            if ( $data['type'] == 'checkbox' ) {
                $emailText = '';
                foreach ( $data['options'] as $option ) {
                    if( $this->submissions[$option['name']] ) {
                        $text = esc_html__('Checked', 'whistleblowing-system');
                    } else {
                        $text = esc_html__('Not checked', 'whistleblowing-system');
                    }
                    $emailText .= $option['miniLabel'] . ": " . $text . "\n";
                }
                $default_label = ( $data['label'] == "" ) ? "Checkbox": $data['label'];
                $body = str_replace("{".$default_label."}", $emailText, $body);
                $all_data .= "<p>".$emailText."</p>";
            } elseif( $data['type'] == 'fullName' ) {
                $val = $this->submissions[$data['name']];
                $fullName = $val['firstName'] . ' ' . $val['middleName'] . ' ' . $val['lastName'];
                $body = str_replace("{" . $data['label'] . "}", $data['label'] . ": " . $fullName, $body);
                $all_data .= $data['label'] . ": " . $fullName;
            } elseif( $data['type'] == 'DateTime' ) {
                $val = $this->submissions[$data['name']];

                $dateTime = $val['time'];
                /* TODO need to optimize solution */
                $date = $val['date'] ? $val['date'] : 0;
                $day = $val['day'] ? $val['day'] : 0;
                $month = $val['month'] ? $val['month'] : 0;
                $year = $val['year'] ? $val['year'] : 0;
                if( $date ) {
                    $dateTime = $val['date'].' '.$val['time'];
                } elseif(  $day ||  $month ||  $year ) {
                    $dateTime = $val['day'].'/'.$val['month'].'/'.$val['year'].' '.$val['time'];
                }
                $body = str_replace("{" . $data['label'] . "}", $data['label'] . ": " . $dateTime, $body);
                $all_data .= "<p>".$data['label'] . ": " . $dateTime."</p>";
            } elseif( $data['type'] == 'address' ) {
                $val = $this->submissions[$data['name']];
                $addressJoined = '';
                if( is_array($val) ) {
                    foreach ($val as $a) {
                        if ($a != '') {
                            $addressJoined .= $a . ', ';
                        }
                    }
                    $addressJoined = substr($addressJoined, 0, -2);
                }

                $body = str_replace("{" . $data['label'] . "}", $data['label'] . ": " . $addressJoined, $body);
                $all_data .= "<p>".$data['label'] . ": " . $addressJoined."</p>";
            } else {
                if( $is_subject ) {
                    $body = str_replace("{" . $data['label'] . "}", $this->submissions[$data['name']], $body);
                } else {
                    $body = str_replace("{" . $data['label'] . "}", $data['label'] . ": " . $this->submissions[$data['name']], $body);
                    $all_data .= "<p>".$data['label'] . ": " . $this->submissions[$data['name']]."</p>";
                }
            }
        }

        $body = str_replace(
            ['<p>{All}</p>', '{All}'],
            $all_data,
            $body
        );
        return $body;
    }
}
