(function ($) {
    'use strict';

    if (typeof WBLS_CONDITIONS !== 'object') return;

    Object.keys(WBLS_CONDITIONS).forEach(function (formId) {
        initForm(formId, WBLS_CONDITIONS[formId]);
    });

    function initForm(formId, config) {
        if (!config || !config.conditions || !config.fields) return;

        const rules  = config.conditions;
        const fields = config.fields;
        const $form  = $('#wbls-form-' + formId);

        if (!$form.length) return;

        if ($form.data('wbls-conditions-init')) return;
        $form.data('wbls-conditions-init', true);

        // Helper to get field config for a field ID (handles compound fields)
        function getFieldConfig(fieldId) {
            // Check if it's a direct field
            if (fields[fieldId]) {
                return fields[fieldId];
            }

            // Check if it's a sub-field of a compound field (e.g., "9_l", "12_street")
            // Extract the parent field ID (remove suffix after last underscore)
            const parts = fieldId.split('_');
            if (parts.length > 1) {
                // Try to find parent field (e.g., "9" from "9_l")
                const parentId = parts[0];
                if (fields[parentId]) {
                    return {
                        ...fields[parentId],
                        // Mark this as a sub-field
                        isSubField: true,
                        subFieldId: fieldId
                    };
                }
            }

            return null;
        }

        // Helper to get field value (handles compound fields)
        function getFieldValue(field, name) {
            if (!field) return '';

            const $el = $form.find('[name="' + name + '"]');

            // Handle rating fields
            if (field.type === 'rating') {
                const $ratingContainer = $el.closest('.wbls-rating');
                if ($ratingContainer.length && !$ratingContainer.is(':visible')) {
                    return '';
                }
                const $checked = $form.find('[name="' + name + '"]:checked');
                return $checked.length ? $checked.val() || '' : '';
            }

            // For compound field sub-fields, check parent visibility
            if (field.isSubField) {
                // Find the parent field row
                const parentFieldId = field.subFieldId.split('_')[0];
                const $parentRow = $form.find('.wblsform-row[data-field-id="' + parentFieldId + '"]');
                if ($parentRow.length && !$parentRow.is(':visible')) {
                    return field.type === 'checkbox' ? false : '';
                }
            } else if (!$el.is(':visible')) {
                // For regular fields
                return field.type === 'checkbox' ? false : '';
            }

            switch (field.type) {
                case 'checkbox':
                    return $el.is(':checked');

                case 'radio':
                case 'anonymous':
                    return $form.find('[name="' + name + '"]:checked').val() || '';

                default:
                    return $el.val() || '';
            }
        }

        function checkCondition(value, condition, expected, fieldType) {
            // Special handling for checkbox
            if (fieldType === 'checkbox') {
                if (expected === "1") {
                    switch (condition) {
                        case 'is': return value === true;
                        case 'is not': return value !== true;
                        case 'empty': return value === false;
                        case 'not empty': return value === true;
                        default: return false;
                    }
                } else {
                    switch (condition) {
                        case 'is': return value === false;
                        case 'is not': return value !== false;
                        case 'empty': return value === true;
                        case 'not empty': return value === false;
                        default: return false;
                    }
                }
            }

            const stringValue = String(value);
            const stringExpected = String(expected);

            switch (condition) {
                case 'is':
                    return stringValue === stringExpected;
                case 'is not':
                    return stringValue !== stringExpected;
                case 'empty':
                    return stringValue === '' || value === false;
                case 'not empty':
                    return stringValue !== '' && value !== false;
                default:
                    return false;
            }
        }

        function evaluate() {
            Object.keys(rules).forEach(function (fieldId) {
                const fieldConfig = rules[fieldId];
                if (!fieldConfig || !fieldConfig.conditions) return;

                let matched = false;

                if (Array.isArray(fieldConfig.conditions)) {
                    fieldConfig.conditions.forEach(function (group) {
                        if (Array.isArray(group)) {
                            let groupMatched = true;

                            group.forEach(function (rule) {
                                if (!groupMatched) return;

                                // Use new helper function to get field config
                                const field = getFieldConfig(rule.field_id);

                                if (!field) {
                                    console.warn('Field not found for ID:', rule.field_id, 'Rule:', rule);
                                    groupMatched = false;
                                    return;
                                }

                                const value = getFieldValue(field, rule.field_name);
                                const ruleMatched = checkCondition(
                                    value,
                                    rule.condition,
                                    rule.value,
                                    field.type
                                );

                                if (!ruleMatched) {
                                    groupMatched = false;
                                }
                            });

                            if (groupMatched) {
                                matched = true;
                            }
                        }
                    });
                }

                const $row = $form.find(
                    '.wblsform-row[data-field-id="' + fieldId + '"]'
                );

                if ($row.length) {
                    const showField = fieldConfig.showField === "1" ||
                        fieldConfig.showField === true ||
                        fieldConfig.showField === 1;

                    if (showField) {
                        matched ? $row.show() : $row.hide();
                    } else {
                        matched ? $row.hide() : $row.show();
                    }
                }
            });
        }

        // Event handlers
        $form.on('input change', 'input, textarea, select', evaluate);
        $form.on('click', '.wbls-rating label', function() {
            setTimeout(evaluate, 10);
        });

        $(document).on('click', '.wbls-new-case-button', evaluate);

        // Initial evaluation
        setTimeout(evaluate, 100);

        // Expose function
        window['wbls_condition_' + formId] = evaluate;
        window['WBLS_Evaluate_' + formId] = evaluate;
    }

    // Handle dynamically loaded forms
    $(document).ready(function() {
        setTimeout(function() {
            if (typeof WBLS_CONDITIONS === 'object') {
                Object.keys(WBLS_CONDITIONS).forEach(function (formId) {
                    const $form = $('#wbls-form-' + formId);
                    if ($form.length && !$form.data('wbls-conditions-init')) {
                        initForm(formId, WBLS_CONDITIONS[formId]);
                    }
                });
            }
        }, 500);
    });

    $(document).on('click', '.wbls-new-case-button', function() {
        setTimeout(function() {
            if (typeof WBLS_CONDITIONS === 'object') {
                Object.keys(WBLS_CONDITIONS).forEach(function (formId) {
                    const $form = $('#wbls-form-' + formId);
                    if ($form.length && !$form.data('wbls-conditions-init')) {
                        initForm(formId, WBLS_CONDITIONS[formId]);
                    }
                });
            }
        }, 300);
    });

})(jQuery);