<?php
namespace WBLS_WhistleBlower\Free;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

class WBLS_frontend {

    private $form_id;
    private $settings;
    private $whistleblower_active = 0;

    private $post_data;

    private $form_content;
    private $form_type;

    private static $queued_footer_forms = [];
    private static $footer_hook_added = false;

    private static $conditions_data = [];
    private static $conditions_script_enqueued = false;

    public function __construct( $attr )
    {
        $this->form_id = $attr['id'];

        $this->form_type = $attr['type'] ?? '';

        $this->settings = get_post_meta($this->form_id, 'wbls_form_settings', 1);
        if ( ! empty($this->settings) && ! isset( $this->settings['enable_chat_upload'] ) ) {
            $this->settings['enable_chat_upload'] = "1";
        }

        $active_theme_id = $attr['theme_id'] ?? 0;
        if( $active_theme_id ) {
            $active_theme = intval($active_theme_id);
        } else {
            $default_theme = get_option('wbls_theme_default');
            $active_theme = isset($this->settings['active_theme']) ? $this->settings['active_theme'] : $default_theme;
        }

        $this->post_data = get_post($this->form_id);
        if (!$this->post_data) {
            return;
        }
        $this->form_content = get_post_meta($this->form_id, 'wbls_form_content', 1);
        $this->whistleblower_active = $this->settings['whistleblower_active'] ?? 1;
        $success_msg = isset($this->settings['success_message']) ? esc_html($this->settings['success_message']) : esc_html__('Form Successfully Submitted', 'whistleblowing-system');
        $error_msg = isset($this->settings['error_message']) ? esc_html($this->settings['error_message']) : esc_html__('Something went wrong', 'whistleblowing-system');
        $success_msg_copy_token = $this->settings['success_message_copy_token'] ?? esc_html__('Please copy and retain this token for future login and for follow-up on the response.', 'whistleblowing-system');
        $show_form_after_submit = $this->settings['wbls_show_form_after_submit'] ?? 1;

        $file_max_size = isset($this->settings['file_max_size']) ? floatval($this->settings['file_max_size']) : 10;

        $form_conditions = get_post_meta($this->form_id, 'wbls_form_conditions', true);
        $field_options   = get_post_meta($this->form_id, 'wbls_field_options', true);

        if ( ! empty($form_conditions) ) {

            // Collect per-form data
            self::$conditions_data[ $this->form_id ] = [
                    'conditions' => $form_conditions,
                    'fields'     => $field_options,
            ];

            // Enqueue JS ONCE
            if ( ! self::$conditions_script_enqueued ) {
                wp_enqueue_script(
                        'wbls-conditions',
                        WBLS_URL . '/frontend/assets/js/conditions.js',
                        ['jquery'],
                        WBLS_VERSION,
                        true
                );

                self::$conditions_script_enqueued = true;
            }

            // Localize ONCE in footer
            add_action('wp_footer', function () {
                if ( ! empty(self::$conditions_data) ) {
                    wp_localize_script(
                            'wbls-conditions',
                            'WBLS_CONDITIONS',
                            self::$conditions_data
                    );
                }
            }, 5);
        }

        $localized_data = array(
            "ajaxnonce" => wp_create_nonce('wbls_ajax_nonce'),
            'ajax_url' => admin_url('admin-ajax.php'),
            'success_msg' => esc_html($success_msg),
            'error_msg' => esc_html($error_msg),
            'token_msg' => esc_html($success_msg_copy_token),
            'file_size_msg' => esc_html__("File size should be less then", 'whistleblowing-system'),
            'file_type_msg' => esc_html__("Invalid file type: allowed types are", 'whistleblowing-system'),
            'file_max_size' => $file_max_size,
            'file_types' => $this->settings['file_types'] ?? ['jpg', 'png', 'gif'],
            'recaptcha_active' => 0,
            'assets_url' => WBLS_URL . '/frontend/assets',
            'show_form_after_submit' => $show_form_after_submit,
        );
        $recaptcha = WBLSLibrary::is_recaptcha_active($this->form_id);
        if (!empty($recaptcha)) {
            $localized_data['recaptcha_active'] = 1;
            $localized_data['recaptcha_version'] = $recaptcha['version'];
            if ($recaptcha['version'] == 'v3') {
                $localized_data['recaptcha_key'] = $recaptcha['reCAPTCHA_v3_site_key'];
            } else {
                $localized_data['recaptcha_key'] = $recaptcha['reCAPTCHA_v2_site_key'];
            }
        }


        
        if ( ! wp_script_is( 'wbls-script', 'registered' ) ) {
            wp_register_script('wbls-script', WBLS_URL . '/frontend/assets/js/script.js', array('jquery'), WBLS_VERSION, true);
        }
        wp_enqueue_script('wbls-script');
        wp_localize_script('wbls-script', 'wbls_front', $localized_data);
        $wp_upload_dir = wp_upload_dir();
        $wbls_style_dir = $wp_upload_dir[ 'basedir' ] . '/wbls-system/wbls-theme-style_' . $active_theme . '.css';
        $wbls_style_url = $wp_upload_dir[ 'baseurl' ] . '/wbls-system/wbls-theme-style_' . $active_theme . '.css';
        wp_enqueue_style(WBLS_PREFIX . '-style', WBLS_URL . '/frontend/assets/css/style.css', array(), WBLS_VERSION);

        if( file_exists($wbls_style_dir) ) {
            wp_enqueue_style(WBLS_PREFIX . '-theme-style_' . $active_theme, $wbls_style_url, array(), WBLS_VERSION);
        } else {
            wp_enqueue_style(WBLS_PREFIX . '-theme-style', WBLS_URL . '/frontend/assets/css/default.css', array(), WBLS_VERSION);
        }

        if ( ! wp_doing_ajax() ) {
            add_action("wp_footer", [$this, 'render_template']);
        }
    }

    public function render_template() {
        // Ensure settings is always an array even if empty
        $template_vars = [
                'settings' => is_array($this->settings) ? $this->settings : [],
        ];

        // Extract variables for use in template
        extract($template_vars);

        // Include the template
        if (file_exists(WBLS_DIR . '/frontend/templates.php')) {
            require_once WBLS_DIR . '/frontend/templates.php';
        }
    }

    private static function queue_footer_form( $html ) {
        self::$queued_footer_forms[] = $html;

        if ( ! self::$footer_hook_added ) {
            add_action('wp_footer', [__CLASS__, 'print_footer_forms'], 100);
            self::$footer_hook_added = true;
        }
    }

    public static function print_footer_forms() {
        if ( ! empty(self::$queued_footer_forms) ) {
            echo wp_kses(implode("\n", self::$queued_footer_forms), WBLSLibrary::$wp_kses_form);
        }
    }

    public function display() {

        if ( is_singular( 'wbls_form' ) ) {
            $wbls_preview_type = sanitize_text_field( get_query_var('wbls_preview_type') );
            if ( $wbls_preview_type != '' && ( $wbls_preview_type === 'form' || $wbls_preview_type === 'tab' ) ) {
                $this->form_type = $wbls_preview_type;
            }
        }
        ob_start();
        if( $this->whistleblower_active ) {
            $buttons = [
                'new_case' => !empty($this->settings['new_case']) ? $this->settings['new_case'] : 'Report a new case',
                'follow_case' => !empty($this->settings['follow_case']) ? $this->settings['follow_case'] : 'Follow up on a case',
                'login_case' => !empty($this->settings['login_case']) ? $this->settings['login_case'] : 'Login',
                'copy_token' => !empty($this->settings['copy_token']) ? $this->settings['copy_token'] : 'Copy token',
                'reply_button' => !empty($this->settings['reply_button']) ? $this->settings['reply_button'] : 'Send',
            ];
            if( $this->form_type == '' ) {
                ?>
                <div class="wbls-front-buttons-container">
                    <span class="wbls-new-case-button"><?php echo esc_html($buttons['new_case']); ?></span>
                    <span class="wbls-followup-button"><?php echo esc_html($buttons['follow_case']); ?></span>
                    <span></span>
                </div>
                <?php

                // Capture form HTML instead of echoing it here
                ob_start();
                $this->form( $buttons );
                $form_html = ob_get_clean();

                // Queue it for printing before </body>
                self::queue_footer_form( $form_html );
            } elseif ( $this->form_type == 'form' ) {
                $this->embed_form();
            } elseif ( $this->form_type == 'login' ) {
                $this->login_form( $buttons );
            } elseif ( $this->form_type == 'tab' ) {
                $this->tab_view( $buttons );
            }
        } else {
            $this->embed_form();
        }
        return ob_get_clean();
    }

    public function embed_form() {
        ?>
        <div class="wbls-form-container wbls-embed-form">
            <form method="post" class="wbls-form" id="wbls-form-<?php echo intval($this->form_id); ?>" <?php echo WBLS_PRO ? 'enctype="multipart/form-data"' : ''; ?>>
                <?php
                if( isset($this->settings['show_form_header']) && $this->settings['show_form_header'] ) {
                    $form_header_text = $this->settings['form_header'];
                    if( !empty($form_header_text) ) {
                    ?>
                        <div class="wbls-front-header">
                           <?php echo wp_kses($form_header_text, WBLSLibrary::$wp_kses_default); ?>
                        </div>
                <?php
                    }
                } ?>
                <input type="hidden" value="<?php echo intval($this->form_id); ?>" name="wbls_form_id">
                <input type="hidden" value="wbls_front_ajax" name="action">
                <input type="hidden" value="wbls_submit_form" name="task">
                <input type="hidden" value="<?php echo intval($this->form_id); ?>" name="wbls_form_id">
                <input type="text" value="" name="wbls_security" class="wbls-security" required>
                <?php echo wp_kses($this->form_content, WBLSLibrary::$wp_kses_form); ?>
            </form>
        </div>
        <?php
    }

    public function login_form( $buttons ) {
        ?>
        <div class="wbls-login-container wbls-embed-login">
            <?php if(isset($this->settings['show_login_header']) && $this->settings['show_login_header']) { ?>
                <?php
                $form_header_text = $this->settings['login_header'];
                if( !empty($form_header_text) ) {
                    echo '<div class="wbls-front-header">';
                    echo wp_kses($form_header_text, WBLSLibrary::$wp_kses_default);
                    echo '</div>';
                }
                ?>
            <?php } ?>
            <div class="wbls-embed-chat-container">
                <div class="wbls-chat-header wbls-embed-chat-header wbls-hidden">
                    <div class="wbls-chat-header-title">
                        <div class="wbls-chat-header-title-text"><?php esc_html_e('Messages', 'whistleblowing-system'); ?></div>
                        <span class="wbls-chat-status"></span>
                    </div>
                    <div class="wbls-chat-header-close" title="<?php esc_attr_e('Log Out', 'whistleblowing-system'); ?>"></div>
                </div>
                <div class="wbls-token-row">
                    <div>
                        <input type="text" value="" name="wbls_token" class="wbls-token-input" autocomplete="off">
                        <input type="text" value="" name="wbls_security" class="wbls-security" required>
                        <button class="wbls-login-button"><?php echo esc_html($buttons['login_case']); ?></button>
                    </div>
                    <span class="wbls-error-msg wbls-hidden"></span>
                </div>
            </div>
        </div>
        <?php
    }

    public function tab_view( $buttons ) {
        ob_start();
        ?>
        <div class="wbls-form-tab-container">
            <div class="wbls-form-tabs">
                <span class="wbls-form-tab wbls-form-tab-active wbls-form-tab-form" data-content="wbls-form-tab-form-content"><?php echo esc_html($buttons['new_case']); ?></span>
                <span class="wbls-form-tab wbls-form-tab-follow" data-content="wbls-form-tab-follow-content"><?php echo esc_html($buttons['follow_case']); ?></span>
            </div>
            <div class="wbls-form-tabs-content">
                <div class="wbls-form-tab-form-content wbls-form-tab-content-active">
                    <?php $this->embed_form(); ?>
                </div>
                <div class="wbls-form-tab-follow-content">
                    <?php $this->login_form( $buttons ); ?>
                </div>
            </div>
        </div>
        <?php
        echo ob_get_clean();
    }

    public function form($buttons) {
        $success_msg_copy_token = $this->settings['success_message_copy_token'] ?? esc_html__('Please copy and retain this token for future login and for follow-up on the response.', 'whistleblowing-system');
        ?>
        <div class="wbls-front-layout<?php echo $this->whistleblower_active ? ' wbls-hidden': ''?>"></div>
        <div class="wbls-front-content wbls-front-form-content wbls-hidden">
            <div class="wbls-form-container wbls-hidden">
                <span class="wbls-front-content-close"></span>
                <div class="wbls-front-header">
                    <div>
                    <?php if(isset($this->settings['show_form_header']) && $this->settings['show_form_header']) { ?>
                    <?php echo wp_kses($this->settings['form_header'], WBLSLibrary::$wp_kses_default); ?>
                    <?php } ?>
                    </div>
                </div>

                <form method="post" class="wbls-form" id="wbls-form-<?php echo intval($this->form_id); ?>" <?php echo WBLS_PRO ? 'enctype="multipart/form-data"' : ''; ?>>    
                    <input type="hidden" value="<?php echo intval($this->form_id); ?>" name="wbls_form_id">
                    <input type="hidden" value="wbls_front_ajax" name="action">
                    <input type="hidden" value="wbls_submit_form" name="task">
                    <input type="hidden" value="<?php echo intval($this->form_id); ?>" name="wbls_form_id">
                    <input type="text" value="" name="wbls_security" class="wbls-security" required>
                    <?php echo wp_kses($this->form_content, WBLSLibrary::$wp_kses_form); ?>
                    <div class="wbls-token-container wbls-hidden">
                        <div class="wbls-token-row">
                            <?php if(isset($this->settings['show_token_header']) && $this->settings['show_token_header']) { ?>
                                <div class="wbls-front-token-header">
                                    <?php echo wp_kses($this->settings['token_header'], WBLSLibrary::$wp_kses_default); ?>
                                </div>
                            <?php } ?>
                            <span class="wbls-token-description">
                            <?php echo esc_html($success_msg_copy_token); ?>
                            </span>

                            <div class="wbls_copy_section">
                                <span class="wbls-token-value"></span>
                                <span class="wbls-copy-button">
                                    <span class="wbls-form-token-copy-tooltip"><?php esc_html_e('Copied', 'whistleblowing-system') ?></span>
                                </span>
                            </div> 
                           
                        </div>
                    </div>
                </form>
                
            </div>
        </div>
        <?php
    }
}