<?php
/**
 * Main plugin class.
 *
 * @package whistleblowing-system
 */

namespace WBLS_WhistleBlower\Free;

defined( 'ABSPATH' ) || die( 'Access Denied' );

final class WBLS_WhistleBlower {

    /**
     * The single instance of the class.
     *
     * @var \WBLS_WhistleBlower\Free\WBLS_WhistleBlower|null
     */
    protected static $instance = null;

    /**
     * Main WBLS_WhistleBlower Instance.
     *
     * Ensures only one instance is loaded or can be loaded.
     *
     * @static
     * @return WBLS_WhistleBlower Main instance.
     */
    public static function instance() {
        if ( is_null( self::$instance ) ) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    /**
     * WBLS_WhistleBlower constructor.
     */
    public function __construct() {
        require_once WBLS_DIR . '/Apps/class-logger.php';

        if ( WBLS_PRO ) {
            require_once WBLS_DIR . '/Apps/rest_api.php';
        }

        // Free started.
        require_once WBLS_DIR . '/admin/includes/pro_trial.php';
        // Free end.

        require_once WBLS_DIR . '/admin/includes/rate_notice.php';

        require_once WBLS_DIR . '/Apps/class-encryption.php';
        require_once WBLS_DIR . '/library.php';

        $this->add_actions();
    }


    /**
     * Activation wrapper for register_activation_hook.
     *
     * @return void
     */
    public static function activate() {
        self::instance()->global_activate();
    }

    /**
     * Deactivation wrapper for register_deactivation_hook.
     *
     * @return void
     */
    public static function deactivate() {
        self::instance()->clear_schedule_event();
    }


    /**
     * Add WordPress actions, filters, shortcodes and hooks.
     *
     * @return void
     */
    private function add_actions() {
        add_action( 'init', array( $this, 'init' ) );
        add_action( 'plugins_loaded', array( $this, 'wbls_plugins_loaded' ) );
        add_action( 'admin_init', array( $this, 'admin_init' ) );
        add_action( 'admin_menu', array( $this, 'admin_menu' ) );

        // Register scripts/styles.
        add_action( 'wp_enqueue_scripts', array( $this, 'register_frontend_scripts' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'register_admin_scripts' ) );

        // AJAX.
        add_action( 'wp_ajax_wbls_admin_ajax', array( $this, 'wbls_admin_ajax' ) );
        add_action( 'wp_ajax_wbls_front_ajax', array( $this, 'wbls_front_ajax' ) );
        add_action( 'wp_ajax_nopriv_wbls_front_ajax', array( $this, 'wbls_front_ajax' ) );

        // Shortcodes.
        add_shortcode( 'wbls-whistleblower-form', array( $this, 'wbls_shortcode' ) );
        add_shortcode( 'wblsform', array( $this, 'wbls_shortcode' ) );

        // Admin footer (pro banner).
        add_action( 'admin_footer', array( $this, 'pro_banner' ) );

        if ( ! WBLS_PRO ) {
            $plugin_basename = plugin_basename( WBLS_DIR . '/whistleblowing.php' );
            add_filter( 'plugin_action_links_' . $plugin_basename, array( $this, 'plugin_action_links' ) );
            add_action( 'wp_ajax_wbls_send_deactivation_reason', array( $this, 'wbls_send_deactivation_reason' ) );
            add_action( 'current_screen', array( $this, 'check_plugins_page' ) );
        }

        require_once WBLS_DIR . '/Apps/blocks.php';

        // Schedule functionality to clear logs.
        add_action(
            'wbls_purge_old_logs_event',
            array( \WBLS_WhistleBlower\Free\WBLS_Logger::class, 'wbls_purge_old_logs' )
        );

        /* Argument registerd to get in preview frontend */
        add_filter( 'query_vars', function( $vars ) {
            $vars[] = 'wbls_preview_type';
            return $vars;
        });
    }

    /**
     * Handle deactivation reason submission via AJAX.
     *
     * @return void
     */
    public function wbls_send_deactivation_reason() {
        $nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';

        if ( ! wp_verify_nonce( $nonce, 'wbls_ajax_nonce' ) ) {
            wp_send_json_error(
                array(
                    'message' => 'Something went wrong.',
                )
            );
        }

        $task = isset( $_POST['task'] ) ? sanitize_text_field( wp_unslash( $_POST['task'] ) ) : '';

        $this->check_plugins_page( 'plugins', $task );
    }

    /**
     * Check if current screen is plugins page and load deactivation modal.
     *
     * @param string|object $current_screen Current screen or screen id.
     * @param string        $task           Optional task.
     *
     * @return void
     */
    public function check_plugins_page( $current_screen, $task = '' ) {
        $cid = '';

        if ( isset( $current_screen->id ) ) {
            $cid = $current_screen->id;
        } elseif ( 'plugins' === $current_screen ) {
            $cid = 'plugins';
        }

        if ( 'plugins' === $cid ) {
            require_once WBLS_DIR . '/Apps/deactivate/deactivate.php';
            new \WBLS_WhistleBlower\Free\WBLS_Deactivate( $task );
        }
    }

    /**
     * Global activation callback.
     * Handles safe plugin activation by scheduling setup tasks for later execution.
     *
     * @since 1.4.2
     * @param bool|mixed $networkwide Network-wide activation flag.
     * @return void
     */
    public function global_activate( $networkwide = false ) { // phpcs:ignore VariableAnalysis.CodeAnalysis.VariableAnalysis.UnusedVariable
        update_option( 'wbls-plugin-version', WBLS_VERSION );

        // Schedule initial setup to run after WordPress is fully loaded
        add_option( 'wbls_initial_setup_pending', true );

        // Only safe operations during activation:
        \WBLS_WhistleBlower\Free\WBLS_Logger::maybe_create_table();

        if ( ! class_exists( '\WBLS_WhistleBlower\Free\WBLS_Rate_Notice' ) ) {
            require_once WBLS_DIR . '/admin/includes/rate_notice.php';
        }
        \WBLS_WhistleBlower\Free\WBLS_Rate_Notice::set_install_date();

        // Schedule rewrite flush for later
        add_option( 'wbls_flush_rewrite_rules', true );
    }

    /**
     * Create default forms on first install.
     *
     * @return void
     */
    public function create_default_forms() {
        if ( ! get_option( 'wbls_default_forms_created', false ) ) {
            require_once WBLS_DIR . '/Apps/defaultForms.php';
            new \WBLS_WhistleBlower\Free\WBLS_DefaultForms();
        }
    }
    public function wbls_generate_encryption_keys() {

        if ( ! get_option('wbls_encryption_key') ) {
            add_option('wbls_encryption_key', bin2hex(random_bytes(32))); // 64-char hex key
        }

        if ( ! get_option('wbls_encryption_iv') ) {
            add_option('wbls_encryption_iv', substr(bin2hex(random_bytes(16)), 0, 16)); // 16-char IV
        }
    }

    /**
     * Plugin action links.
     *
     * Adds action links to the plugin list table.
     *
     * Fired by `plugin_action_links` filter.
     *
     * @since 1.1.1
     * @access public
     *
     * @param array $links An array of plugin action links.
     *
     * @return array An array of plugin action links.
     */
    public function plugin_action_links( $links ) {
        $links['go_pro'] = sprintf(
            '<a href="%1$s" target="_blank" class="wbls-plugins-gopro">%2$s</a>',
            'https://whistleblowing-form.de/en/produkt/whistleblowing-system-starter/?from=plugin',
            esc_html__( 'Get Whistleblower Pro', 'whistleblowing-system' )
        );

        return $links;
    }

    /**
     * Output pro banner in admin footer.
     *
     * @return void
     */
    public function pro_banner() {
        require_once WBLS_DIR . '/admin/wistleblower_templates.php';
    }

    /**
     * Admin initialization hook.
     *
     * @return void
     */
    public function admin_init() {
        $page = isset( $_GET['page'] ) ? sanitize_text_field( wp_unslash( $_GET['page'] ) ) : '';

        if ( 'whistleblower_theme_edit' === $page ) {
            require_once WBLS_DIR . '/admin/controllers/ControllerThemes.php';

            $ob = new \WBLS_WhistleBlower\Free\WBLS_ControllerThemes();
            $ob->init();
        }

        $version = get_option( 'wbls-plugin-version', false );

        if ( ! $version || version_compare( $version, '1.3.11', '<' ) ) {
            WBLSLibrary::wbls_migrate_form_structure();
        }

        if ( ! get_option( 'wbls_install_date' ) ) {
            update_option( 'wbls_install_date', time() );
        }
    }

    /**
     * Plugins loaded hook handler.
     *
     * @return void
     */
    public function wbls_plugins_loaded() {
        $version = get_option( 'wbls-plugin-version', false );

        // Create logs table.
        if ( ! $version || version_compare( $version, '1.3.15', '<' ) ) {
            \WBLS_WhistleBlower\Free\WBLS_Logger::maybe_create_table();
            update_option( 'wbls_plugin_version', WBLS_VERSION );
        }
    }

    /**
     * Shortcode callback.
     *
     * @param array $attr Shortcode attributes.
     *
     * @return string
     */
    public function wbls_shortcode( $attr ) {
        require_once WBLS_DIR . '/frontend/frontend.php';

        if ( ! isset( $attr['id'] ) ) {
            $old_form_id = get_option( 'wbls-oldForm_id' );

            if ( $old_form_id ) {
                $attr = array(
                    'id' => (int) $old_form_id,
                );
            }
        }

        $ob = new \WBLS_WhistleBlower\Free\WBLS_frontend( $attr );

        return $ob->display();
    }

    /**
     * Frontend AJAX handler.
     *
     * @return void
     */
    public function wbls_front_ajax() {
        $nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';

        if ( ! wp_verify_nonce( $nonce, 'wbls_ajax_nonce' ) ) {
            die( esc_html__( 'Security check', 'whistleblowing-system' ) );
        }

        $task = isset( $_POST['task'] ) ? sanitize_text_field( wp_unslash( $_POST['task'] ) ) : '';

        require_once WBLS_DIR . '/frontend/Controller.php';

        new \WBLS_WhistleBlower\Free\WBLSFront_Controller( $task );
    }

    /**
     * Admin AJAX handler.
     *
     * @return void
     */
    public function wbls_admin_ajax() {
        $nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';

        if ( ! wp_verify_nonce( $nonce, 'wbls_ajax_nonce' ) ) {
            die( esc_html__( 'Security check', 'whistleblowing-system' ) );
        }

        $task = isset( $_POST['task'] ) ? sanitize_text_field( wp_unslash( $_POST['task'] ) ) : '';

        require_once WBLS_DIR . '/admin/controllers/Controller.php';

        new \WBLS_WhistleBlower\Free\WBLS_Controller( $task );
    }

    /**
     * Register admin scripts and styles.
     *
     * @param string $hook Current admin page hook.
     *
     * @return void
     */
    public function register_admin_scripts( $hook ) {
        wp_register_style(
            WBLS_PREFIX . '-settings',
            WBLS_URL . '/admin/assets/css/settings.css',
            array(),
            WBLS_VERSION
        );

        wp_register_script(
            WBLS_PREFIX . '-settings',
            WBLS_URL . '/admin/assets/js/settings.js',
            array( 'jquery' ),
            WBLS_VERSION
        );

        wp_register_style(
            WBLS_PREFIX . '-style',
            WBLS_URL . '/admin/assets/css/style.css',
            array(),
            WBLS_VERSION
        );

        wp_register_script(
            WBLS_PREFIX . '-conditions',
            WBLS_URL . '/admin/assets/js/conditions.js',
            array( 'jquery' ),
            WBLS_VERSION
        );

        wp_register_style(
            WBLS_PREFIX . '-edit',
            WBLS_URL . '/admin/assets/css/edit.css',
            array(),
            WBLS_VERSION
        );

        wp_enqueue_editor();

        wp_register_script(
            WBLS_PREFIX . '-edit',
            WBLS_URL . '/admin/assets/js/edit.js',
            array( 'jquery', 'jquery-ui-draggable', 'wbls-conditions', 'wp-editor' ),
            WBLS_VERSION
        );

        wp_register_script(
            WBLS_PREFIX . '-select2',
            WBLS_URL . '/admin/assets/js/select2.min.js',
            array( 'jquery' ),
            WBLS_VERSION
        );

        wp_register_style(
            WBLS_PREFIX . '-select2',
            WBLS_URL . '/admin/assets/css/select2.min.css',
            array(),
            WBLS_VERSION
        );

        wp_register_style(
            WBLS_PREFIX . '-themes',
            WBLS_URL . '/admin/assets/css/themes.css',
            array(),
            WBLS_VERSION
        );

        wp_register_script(
            WBLS_PREFIX . '-themes',
            WBLS_URL . '/admin/assets/js/themes.js',
            array( 'jquery', 'wp-color-picker' ),
            WBLS_VERSION
        );

        wp_register_style(
            WBLS_PREFIX . '-submissions',
            WBLS_URL . '/admin/assets/css/submissions.css',
            array(),
            WBLS_VERSION
        );

        wp_register_script(
            WBLS_PREFIX . '-submissions',
            WBLS_URL . '/admin/assets/js/submissions.js',
            array( 'jquery' ),
            WBLS_VERSION
        );

        wp_register_script(
            WBLS_PREFIX . '-admin',
            WBLS_URL . '/admin/assets/js/admin.js',
            array( 'jquery' ),
            WBLS_VERSION
        );

        wp_enqueue_script( WBLS_PREFIX . '-admin' );

        wp_localize_script(
            WBLS_PREFIX . '-admin',
            'wbls_admin',
            array(
                'ajaxnonce'           => wp_create_nonce( 'wbls_ajax_nonce' ),
                'form_success_delete' => esc_html__( 'Form successfully deleted', 'whistleblowing-system' ),
                'theme_success_delete' => esc_html__( 'Theme successfully deleted', 'whistleblowing-system' ),
                'form_error_delete'   => esc_html__( 'Something went wrong', 'whistleblowing-system' ),
                'success_save'        => esc_html__( 'Data successfully saved', 'whistleblowing-system' ),
                'page'                => isset( $_GET['page'] ) ? sanitize_text_field( wp_unslash( $_GET['page'] ) ) : '',
            )
        );

        wp_enqueue_style(
            WBLS_PREFIX . '-admin',
            WBLS_URL . '/admin/assets/css/admin.css',
            array(),
            WBLS_VERSION
        );

        // Deactivate scripts.
        if ( 'plugins.php' === $hook && ! WBLS_PRO ) {
            wp_register_script(
                WBLS_PREFIX . '-deactivate',
                WBLS_URL . '/Apps/deactivate/assets/deactivate.js',
                array( 'jquery' ),
                WBLS_VERSION
            );

            wp_localize_script(
                WBLS_PREFIX . '-deactivate',
                'deactivate_options',
                array(
                    'ajax_url' => admin_url( 'admin-ajax.php' ),
                    'nonce'    => wp_create_nonce( 'wbls_ajax_nonce' ),
                    'pro'      => WBLS_PRO,
                )
            );

            wp_register_style(
                WBLS_PREFIX . '-deactivate',
                WBLS_URL . '/Apps/deactivate/assets/deactivate.css',
                array(),
                WBLS_VERSION
            );
        }

        wp_enqueue_style(
            'wbls_logs_css',
            WBLS_URL . '/admin/assets/css/logs.css',
            array(),
            WBLS_VERSION
        );

        wp_enqueue_script(
            'wbls_logs_js',
            WBLS_URL . '/admin/assets/js/logs.js',
            array( 'jquery' ),
            WBLS_VERSION,
            true
        );
    }

    /**
     * Register frontend scripts.
     *
     * @return void
     */
    public function register_frontend_scripts() {
        wp_register_script(
            'wbls-script',
            WBLS_URL . '/frontend/assets/js/script.js',
            array( 'jquery' ),
            WBLS_VERSION,
            true
        );
    }

    /**
     * Initialize plugin and handle scheduled tasks.
     * Registers CPTs, schedules cleanup events, and processes pending setup tasks.
     *
     * @since 1.4.2
     * @return void
     */
    public function init() {
        $this->wbls_register_cpt();
        $this->wbls_schedule_purge_old_logs();

        // Handle initial setup if scheduled during activation
        if ( get_option( 'wbls_initial_setup_pending' ) ) {
            $this->run_initial_setup();
            delete_option( 'wbls_initial_setup_pending' );
        }

        // Handle rewrite rules flush if scheduled
        if ( get_option( 'wbls_flush_rewrite_rules' ) ) {
            flush_rewrite_rules();
            delete_option( 'wbls_flush_rewrite_rules' );
        }

        $this->wbls_generate_encryption_keys();
    }

    /**
     * Run initial plugin setup after WordPress is fully loaded.
     * Handles theme creation, default forms, and other setup tasks safely.
     *
     * @since 1.4.2
     * @return void
     */
    private function run_initial_setup() {
        // Now it's safe to check posts and create content
        $count_themes = wp_count_posts( 'wbls_theme' )->publish;

        if ( ! $count_themes ) {
            require_once WBLS_DIR . '/admin/controllers/ControllerThemes.php';
            $ob = new \WBLS_WhistleBlower\Free\WBLS_ControllerThemes();
            $ob->save_theme( false );
        }

        if ( ! WBLS_PRO ) {
            $this->create_default_forms();
        }
    }

    /**
     * Schedule purge of old logs.
     *
     * @return void
     */
    public function wbls_schedule_purge_old_logs() {
        $hook      = 'wbls_purge_old_logs_event';
        $scheduled = wp_next_scheduled( $hook );

        $row             = get_option( 'wbls_global_settings', '{}' );
        $global_settings = json_decode( is_string( $row ) ? $row : '{}', true );
        $s               = is_array( $global_settings ) ? $global_settings : array();

        $days = isset( $s['logs_lifetime'] ) ? (int) $s['logs_lifetime'] : 30;

        // If you have a toggle like logs_active, honor it; otherwise treat as enabled.
        $active = true;

        if ( array_key_exists( 'logs_active', $s ) ) {
            $active = ! empty( $s['logs_active'] ) && '0' !== $s['logs_active'];
        }

        if ( $active && $days >= 1 ) {
            if ( ! $scheduled ) {
                wp_schedule_event( time() + HOUR_IN_SECONDS, 'daily', $hook );
            }
        } else {
            if ( $scheduled ) {
                wp_clear_scheduled_hook( $hook );
            }
        }
    }

    /**
     * Register admin menu.
     *
     * @return void
     */
    public function admin_menu() {
        $nicename = 'Whistleblower';

        add_menu_page(
            $nicename,
            $nicename,
            'manage_options',
            'whistleblower_forms',
            array( $this, 'admin_pages' ),
            WBLS_URL . '/admin/assets/images/logo.svg'
        );

        add_submenu_page(
            'whistleblower_forms',
            esc_html__( 'All Forms', 'whistleblowing-system' ),
            esc_html__( 'All Forms', 'whistleblowing-system' ),
            'manage_options',
            'whistleblower_forms',
            array( $this, 'admin_pages' )
        );

        add_submenu_page(
            'whistleblower_forms',
            esc_html__( 'Submissions', 'whistleblowing-system' ),
            esc_html__( 'Submissions', 'whistleblowing-system' ),
            'manage_options',
            'whistleblower_submissions',
            array( $this, 'admin_pages' )
        );

        add_submenu_page(
            'whistleblower_forms',
            esc_html__( 'Settings', 'whistleblowing-system' ),
            esc_html__( 'Settings', 'whistleblowing-system' ),
            'manage_options',
            'whistleblower_settings',
            array( $this, 'admin_pages' )
        );

        add_submenu_page(
            'whistleblower_forms_hidden_menu',
            esc_html__( 'Edit', 'whistleblowing-system' ),
            esc_html__( 'Edit', 'whistleblowing-system' ),
            'manage_options',
            'whistleblower_form_edit',
            array( $this, 'admin_pages' )
        );

        add_submenu_page(
            'whistleblower_forms_hidden_menu',
            esc_html__( 'Edit', 'whistleblowing-system' ),
            esc_html__( 'Edit', 'whistleblowing-system' ),
            'manage_options',
            'whistleblower_submission_edit',
            array( $this, 'admin_pages' )
        );

        add_submenu_page(
            'whistleblower_forms_hidden_menu',
            esc_html__( 'Edit', 'whistleblowing-system' ),
            esc_html__( 'Edit', 'whistleblowing-system' ),
            'manage_options',
            'whistleblower_submission_item_edit',
            array( $this, 'admin_pages' )
        );

        add_submenu_page(
            'whistleblower_forms_hidden_menu',
            esc_html__( 'Edit', 'whistleblowing-system' ),
            esc_html__( 'Edit', 'whistleblowing-system' ),
            'manage_options',
            'whistleblower_theme_edit',
            array( $this, 'admin_pages' )
        );

        add_submenu_page(
            'whistleblower_forms',
            esc_html__( 'Themes', 'whistleblowing-system' ),
            esc_html__( 'Themes', 'whistleblowing-system' ),
            'manage_options',
            'whistleblower_themes',
            array( $this, 'admin_pages' )
        );

        add_submenu_page(
            'whistleblower_forms',
            esc_html__( 'Logs', 'whistleblowing-system' ),
            esc_html__( 'Logs', 'whistleblowing-system' ),
            'manage_options',
            'whistleblower_logs',
            array( $this, 'admin_pages' )
        );

        add_submenu_page(
            'whistleblower_forms',
            esc_html__( 'Support', 'whistleblowing-system' ),
            esc_html__( 'Support', 'whistleblowing-system' ),
            'manage_options',
            'wbls_support_redirect',
            '__return_null'
        );

        $support_url = WBLS_PRO
            ? 'https://whistleblowing-form.de/en/contact-whistleblowing-system/'
            : 'https://wordpress.org/support/plugin/whistleblowing-system/';

        // After menu is registered, override its link.
        add_action(
            'admin_head',
            function () use ( $support_url ) {
                global $submenu;

                if ( isset( $submenu['whistleblower_forms'] ) ) {
                    foreach ( $submenu['whistleblower_forms'] as &$item ) {
                        if ( 'wbls_support_redirect' === $item[2] ) {
                            $item[2] = $support_url;
                        }
                    }
                }
            }
        );
    }

    /**
     * Admin pages router.
     *
     * @return void
     */
    public function admin_pages() {
        $page = isset( $_GET['page'] ) ? sanitize_text_field( wp_unslash( $_GET['page'] ) ) : '';

        require_once WBLS_DIR . '/admin/' . $page . '_page.php';

        $class_name = str_replace( '_', ' ', $page );
        $class_name = str_replace( ' ', '', ucwords( $class_name ) );
        $class_name = '\\WBLS_WhistleBlower\\Free\\' . $class_name;

        if ( class_exists( $class_name ) ) {
            new $class_name();
        }
    }

    /**
     * Add custom capabilities to WordPress roles.
     * Grants necessary permissions to Administrator role for plugin CPT management.
     *
     * @since 1.4.2
     * @return void
     */
    private function add_capabilities_to_roles() {
        $admin_role = get_role( 'administrator' );
        if ( $admin_role ) {
            $caps = array( 'wbls_form_subm', 'wbls_theme' );
            foreach ( $caps as $cap ) {
                if ( ! $admin_role->has_cap( $cap ) ) {
                    $admin_role->add_cap( $cap );
                }
            }
        }
    }

    /**
     * Register custom post types.
     *
     * @return void
     */
    public function wbls_register_cpt() {
        // Add capabilities when registering CPTs
        $this->add_capabilities_to_roles();

        register_post_type(
            'wbls_form',
            array(
                'labels'              => array(
                    'name'          => __( 'Forms', 'whistleblowing-system' ),
                    'singular_name' => __( 'Form', 'whistleblowing-system' ),
                ),
                'public'              => true,
                'exclude_from_search' => true,
                'show_menu'           => false,
                'show_ui'             => false,
                'show_in_admin_bar'   => false,
                'show_in_rest'        => true,
                'rewrite'             => false,
                'query_var'           => false,
                'can_export'          => false,
                'supports'            => array( 'title', 'author', 'revisions' ),
                'capability_type'     => 'post',
                'map_meta_cap'        => false,
            )
        );

        $labels = array(
            'name'               => esc_html_x( 'Submissions', 'Post Type General Name', 'whistleblowing-system' ),
            'singular_name'      => esc_html_x( 'Submission', 'Post Type Singular Name', 'whistleblowing-system' ),
            'menu_name'          => esc_html__( 'Submissions', 'whistleblowing-system' ),
            'name_admin_bar'     => esc_html__( 'Submissions', 'whistleblowing-system' ),
            'parent_item_colon'  => esc_html__( 'Parent Item:', 'whistleblowing-system' ),
            'all_items'          => esc_html__( 'All Items', 'whistleblowing-system' ),
            'add_new_item'       => esc_html__( 'Add New Item', 'whistleblowing-system' ),
            'add_new'            => esc_html__( 'Add New', 'whistleblowing-system' ),
            'new_item'           => esc_html__( 'New Item', 'whistleblowing-system' ),
            'edit_item'          => esc_html__( 'Edit Item', 'whistleblowing-system' ),
            'update_item'        => esc_html__( 'Update Item', 'whistleblowing-system' ),
            'view_item'          => esc_html__( 'View Item', 'whistleblowing-system' ),
            'search_items'       => esc_html__( 'Search Item', 'whistleblowing-system' ),
            'not_found'          => '',
            'not_found_in_trash' => esc_html__( 'Not found in Trash', 'whistleblowing-system' ),
        );

        $args = array(
            'label'               => esc_html__( 'Submission', 'whistleblowing-system' ),
            'description'         => esc_html__( 'Form Submissions', 'whistleblowing-system' ),
            'labels'              => $labels,
            'supports'            => false,
            'hierarchical'        => false,
            'public'              => false,
            'show_ui'             => true,
            'show_in_menu'        => false,
            'menu_position'       => 5,
            'show_in_admin_bar'   => false,
            'show_in_nav_menus'   => false,
            'can_export'          => true,
            'has_archive'         => false,
            'exclude_from_search' => true,
            'publicly_queryable'  => false,
            'rewrite'             => false,
            'capabilities'        => array(
                'publish_posts'       => 'wbls_form_subm',
                'edit_posts'          => 'wbls_form_subm',
                'edit_others_posts'   => 'wbls_form_subm',
                'delete_posts'        => 'wbls_form_subm',
                'delete_others_posts' => 'wbls_form_subm',
                'read_private_posts'  => 'wbls_form_subm',
                'edit_post'           => 'wbls_form_subm',
                'delete_post'         => 'wbls_form_subm',
                'read_post'           => 'wbls_form_subm',
            ),
        );

        register_post_type( 'wbls_form_subm', $args );

        $labels = array(
            'name'               => esc_html_x( 'Themes', 'Post Type General Name', 'whistleblowing-system' ),
            'singular_name'      => esc_html_x( 'Theme', 'Post Type Singular Name', 'whistleblowing-system' ),
            'menu_name'          => esc_html__( 'Themes', 'whistleblowing-system' ),
            'name_admin_bar'     => esc_html__( 'Themes', 'whistleblowing-system' ),
            'parent_item_colon'  => esc_html__( 'Parent Item:', 'whistleblowing-system' ),
            'all_items'          => esc_html__( 'All themes', 'whistleblowing-system' ),
            'add_new_item'       => esc_html__( 'Add New theme', 'whistleblowing-system' ),
            'add_new'            => esc_html__( 'Add New', 'whistleblowing-system' ),
            'new_item'           => esc_html__( 'New Item', 'whistleblowing-system' ),
            'edit_item'          => esc_html__( 'Edit Item', 'whistleblowing-system' ),
            'update_item'        => esc_html__( 'Update Item', 'whistleblowing-system' ),
            'view_item'          => esc_html__( 'View Item', 'whistleblowing-system' ),
            'search_items'       => esc_html__( 'Search Item', 'whistleblowing-system' ),
            'not_found'          => '',
            'not_found_in_trash' => esc_html__( 'Not found in Trash', 'whistleblowing-system' ),
        );

        $args = array(
            'label'               => esc_html__( 'Theme', 'whistleblowing-system' ),
            'description'         => esc_html__( 'Form themes', 'whistleblowing-system' ),
            'labels'              => $labels,
            'supports'            => ['title'],
            'hierarchical'        => false,
            'public'              => false,
            'show_ui'             => true,
            'show_in_menu'        => false,
            'menu_position'       => 100,
            'show_in_admin_bar'   => false,
            'show_in_nav_menus'   => true,
            'show_in_rest'        => true,
            'can_export'          => true,
            'has_archive'         => false,
            'exclude_from_search' => true,
            'publicly_queryable'  => false,
            'rewrite'             => false,
            'capabilities'        => array(
                'publish_posts'       => 'wbls_theme',
                'edit_posts'          => 'wbls_theme',
                'edit_others_posts'   => 'wbls_theme',
                'delete_posts'        => 'wbls_theme',
                'delete_others_posts' => 'wbls_theme',
                'read_private_posts'  => 'wbls_theme',
                'edit_post'           => 'wbls_theme',
                'delete_post'         => 'wbls_theme',
                'read_post'           => 'wbls_theme',
            ),
        );

        register_post_type( 'wbls_theme', $args );
    }

    /**
     * Clear scheduled events on deactivation.
     *
     * @return void
     */
    public function clear_schedule_event() {
        if ( WBLS_PRO ) {
            $timestamp = wp_next_scheduled( 'license_check_schedule_action' );

            if ( $timestamp ) {
                wp_unschedule_event( $timestamp, 'license_check_schedule_action' );
            }
        }

        wp_clear_scheduled_hook( 'wbls_purge_old_logs_event' );
        flush_rewrite_rules();
    }
}

